<?php
namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Client;
use App\Models\Plan;
use App\Models\PlanCrsCut;
use App\Models\PlanCtlCut;
use App\Models\PlanProcess;
use App\Models\Scrap;
use DB;
use Illuminate\Http\Request;
use Log;

class SummaryController extends Controller
{
//     public function summary(Request $request)
// {
//     $clients = Client::all();
//     $query = Plan::with(['client', 'coil', 'grade', 'inward'])
//                  ->where('status', 'completed');

//     if ($request->has('client_id') && $request->client_id) {
//         $query->where('client_id', $request->client_id);
//     }

//     $plans = $query->paginate(10);

//     return view('admin.plan.summary', compact('plans', 'clients'));
// }
    public function summary(Request $request)
    {
        $query = Plan::with(['client', 'coil', 'processes.crsCuts', 'processes.ctlCuts'])
            ->where('status', 'completed');

        if ($request->has('client_id') && $request->client_id) {
            $query->where('client_id', $request->client_id);
        }

        $plans = $query->paginate(10);

        return view('admin.plan.summary', compact('plans'));
    }

    public function store(Request $request)
    {
        $request->validate([
            'process'        => 'required|in:crs,ctl',
            'client_id'      => 'required',
            'cut_id'         => 'required|integer',
            'balance_weight' => 'required|numeric|min:0',
            'plan_id'        => 'required|exists:plans,id',
        ]);

        try {
            DB::beginTransaction();

            $originalPlan = Plan::with('coil', 'client')->findOrFail($request->plan_id);

            $newPlan = Plan::create([
                'client_id'         => $originalPlan->client_id,
                'coil_id'           => $originalPlan->coil_id,
                'inward_id'         => $originalPlan->inward_id,
                'grade_id'          => $originalPlan->grade_id,
                'total_used_weight' => 0,
                'total_scrap'       => 0,
                'balance_weight'    => $request->balance_weight,
                'status'            => 'pending',
                'dispatch_status'   => 'pending',
            ]);

            $newProcess = PlanProcess::create([
                'plan_id'      => $newPlan->id,
                'process_type' => $request->process,
            ]);

            $newCutId = null;
            if ($request->process === 'crs') {
                $originalCut = PlanCrsCut::findOrFail($request->cut_id);
                $newCut      = PlanCrsCut::create([
                    'process_id'     => $newProcess->id,
                    'cut_width'      => $originalCut->cut_width,
                    'quantity'       => $originalCut->quantity,
                    'weight'         => $request->balance_weight,
                    'prod_qty'       => 0,
                    'current_weight' => 0,
                    'bal_weight'     => $request->balance_weight,
                    'crs_bal_weight' => $request->balance_weight,
                ]);
                $newCutId = $newCut->crs_cut_id;
                $originalCut->crs_bal_weight -= $request->balance_weight;
                $originalCut->bal_weight -= $request->balance_weight;
                $originalCut->save();
            } else {
                $originalCut = PlanCtlCut::findOrFail($request->cut_id);
                $newCut      = PlanCtlCut::create([
                    'process_id'      => $newProcess->id,
                    'cut_width'       => $originalCut->cut_width,
                    'cut_length'      => $originalCut->cut_length,
                    'quantity'        => $originalCut->quantity,
                    'weight'          => $request->balance_weight,
                    'prod_qty'        => 0,
                    'current_weight'  => 0,
                    'bal_weight'      => $request->balance_weight,
                    'ctl_bal_weight'  => $request->balance_weight,
                    'crs_id'          => $originalCut->crs_id,
                    'dependent_index' => $originalCut->dependent_index,
                ]);
                $newCutId = $newCut->ctl_cut_id;
                $originalCut->ctl_bal_weight -= $request->balance_weight;
                $originalCut->bal_weight -= $request->balance_weight;
                $originalCut->save();
            }

            $originalPlan->balance_weight -= $request->balance_weight;
            $originalPlan->save();

            DB::commit();

            return response()->json([
                'success'        => true,
                'message'        => 'Plan created successfully',
                'new_plan_id'    => $newPlan->id,
                'new_cut_id'     => $newCutId,
                'process'        => $request->process,
                'balance_weight' => $request->balance_weight,
            ]);
        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json(['success' => false, 'message' => $e->getMessage()], 500);
        }
    }

    public function getCrsCut($cutId)
    {
        $cut = PlanCrsCut::find($cutId);
        if (! $cut) {
            return response()->json(['error' => 'CRS cut not found'], 404);
        }
        return response()->json([
            'cut' => [
                'cut_width'      => $cut->cut_width,
                'quantity'       => $cut->quantity,
                'weight'         => $cut->weight,
                'crs_bal_weight' => $cut->crs_bal_weight,
            ],
        ]);
    }

    public function getCtlCut($cutId)
    {
        $cut = PlanCtlCut::find($cutId);
        if (! $cut) {
            return response()->json(['error' => 'CTL cut not found'], 404);
        }
        return response()->json([
            'cut' => [
                'cut_width'       => $cut->cut_width,
                'cut_length'      => $cut->cut_length,
                'quantity'        => $cut->quantity,
                'weight'          => $cut->weight,
                'ctl_bal_weight'  => $cut->ctl_bal_weight,
                'crs_id'          => $cut->crs_id,
                'dependent_index' => $cut->dependent_index,
                'crs_process_id'  => $cut->crs_id ? PlanProcess::find($cut->process_id)->id : null,
                'crs_index'       => $cut->dependent_index ? explode('-', $cut->dependent_index)[1] : null,
            ],
        ]);
    }

    public function scrap(Request $request)
    {
        $request->validate([
            'plan_id'   => 'required|exists:plans,id',
            'cut_id'    => 'required|integer',
            'process'   => 'required|in:crs,ctl',
            'client_id' => 'required|string|max:255',
            'weight'    => 'required|numeric|min:0',
        ]);

        try {
            DB::beginTransaction();

            Log::info('Scrap Request Data: ', $request->all());

            // Convert client_code to client_id
            $client = Client::where('client_code', $request->client_id)->first();
            if (! $client) {
                throw new \Exception('Client not found for code: ' . $request->client_id);
            }
            Log::info('Client Found: ', $client->toArray());

            // Fetch plan
            $plan = Plan::with('coil', 'processes')->findOrFail($request->plan_id);
            Log::info('Plan Fetched: ', $plan->toArray());

            // Validate cut_id and get process_id
            $processId = null;
            if ($request->process === 'crs') {
                $cut = PlanCrsCut::where('crs_cut_id', $request->cut_id)->first();
                if (! $cut) {
                    throw new \Exception('CRS cut not found for cut_id: ' . $request->cut_id);
                }
                Log::info('CrsCut Found: ', $cut->toArray());
                $processId = $cut->process_id;
            } else {
                $cut = PlanCtlCut::where('ctl_cut_id', $request->cut_id)->first();
                if (! $cut) {
                    throw new \Exception('CTL cut not found for cut_id: ' . $request->cut_id);
                }
                Log::info('CtlCut Found: ', $cut->toArray());
                $processId = $cut->process_id;
            }

            // Validate process_id from cut
            if (! $processId) {
                throw new \Exception('No process_id associated with cut_id: ' . $request->cut_id);
            }

            // Verify the process_id exists in plan_processes
            $process = PlanProcess::where('process_id', $processId)
                ->where('plan_id', $request->plan_id)
                ->where('process_type', $request->process)
                ->first();

            if (! $process) {
                Log::info('No PlanProcess found for process_id: ' . $processId . ', plan_id: ' . $request->plan_id . ', process_type: ' . $request->process . '. Attempting to create.');
                try {
                    $process = PlanProcess::create([
                        'plan_id'      => $request->plan_id,
                        'process_type' => $request->process,
                    ]);
                    if (! $process || ! $process->process_id) {
                        throw new \Exception('Failed to create PlanProcess: No process_id generated.');
                    }
                    Log::info('Created new PlanProcess: ', $process->toArray());
                    $processId = $process->process_id;
                } catch (\Exception $e) {
                    throw new \Exception('Failed to create PlanProcess for plan_id: ' . $request->plan_id . ', process_type: ' . $request->process . '. Error: ' . $e->getMessage());
                }
            } else {
                Log::info('Existing PlanProcess found: ', $process->toArray());
            }

            // Validate process_id
            if (! $processId) {
                throw new \Exception('Invalid process_id for PlanProcess: ' . json_encode($process->toArray()));
            }

            // Prepare scrap data with all fields
            $scrapData = [
                'plan_id'         => $request->plan_id,
                'process_id'      => $processId,
                'cut_id'          => $request->cut_id,
                'process_type'    => $request->process,
                'scrap_weight'    => $request->weight,
                'vehicle_details' => null,
                'driver_name'     => null,
                'transport_date'  => null,
                'destination'     => null,
                'status'          => 'pending',
                'created_at'      => now(),
                'updated_at'      => now(),
            ];

            Log::info('Scrap Data Prepared: ', $scrapData);

            // Create scrap record
            $scrap = Scrap::create($scrapData);
            Log::info('Scrap Record Created: ', $scrap->toArray());

            // Update balance weight in crs_cuts or ctl_cuts
            if ($request->process === 'crs') {
                $updated = PlanCrsCut::where('crs_cut_id', $request->cut_id)
                    ->update(['crs_bal_weight' => 0]);
                Log::info('CrsCut Updated: ', ['cut_id' => $request->cut_id, 'updated' => $updated]);
            } else {
                $updated = PlanCtlCut::where('ctl_cut_id', $request->cut_id)
                    ->update(['ctl_bal_weight' => 0]);
                Log::info('CtlCut Updated: ', ['cut_id' => $request->cut_id, 'updated' => $updated]);
            }

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'Weight successfully marked as scrap.',
            ]);
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Scrap Error: ' . $e->getMessage(), ['trace' => $e->getTraceAsString()]);
            return response()->json([
                'success' => false,
                'message' => 'Failed to mark as scrap: ' . $e->getMessage(),
            ], 500);
        }
    }
}
