@extends('layouts.structure')

@section('content')
    <style>
        .container {
            margin-top: 80px;
        }

        .coil-row {
            background-color: #f8f9fa;
            border: 1px solid #dee2e6;
            padding: 15px;
            margin-bottom: 15px;
            border-radius: 5px;
        }

        .coil-row.new {
            background-color: #e6f3ff;
            transition: background-color 1s ease;
        }

        .form-label {
            font-weight: bold;
        }

        .invalid-feedback {
            color: #dc3545;
            font-size: 0.875em;
            display: none;
        }

        .is-invalid~.invalid-feedback {
            display: block;
        }

        .autocomplete-suggestions {
            position: absolute;
            z-index: 1000;
            background: white;
            border: 1px solid #ccc;
            max-height: 200px;
            overflow-y: auto;
            width: 100%;
            box-shadow: 0 2px 4px rgba(0, 0, 0, 0.2);
        }

        .autocomplete-suggestions ul {
            list-style: none;
            padding: 0;
            margin: 0;
        }

        .autocomplete-suggestions li {
            padding: 8px;
            cursor: pointer;
        }

        .autocomplete-suggestions li:hover {
            background-color: #f0f0f0;
        }

        .remove-coil {
            cursor: pointer;
            color: red;
            font-size: 1.2em;
            display: none;
        }

        .alert-dismissible {
            margin-bottom: 1rem;
        }

        .coil-template {
            display: none;
        }
    </style>

    <div class="container">
        <div class="card mb-5">
            <div class="card-header bg-primary text-white">
                <h5 class="mt-2 mb-0">Edit Inward: {{ $inward->inward_code }}</h5>
            </div>
            <div class="card-body p-4">
                @if ($errors->any())
                    <div class="alert alert-danger alert-dismissible fade show" role="alert">
                        <strong>Validation Errors:</strong>
                        <ul class="mb-0">
                            @foreach ($errors->all() as $error)
                                <li>{{ $error }}</li>
                            @endforeach
                        </ul>
                        <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
                    </div>
                @endif
                @if (session('success'))
                    <div class="alert alert-success alert-dismissible fade show" role="alert">
                        {{ session('success') }}
                        <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
                    </div>
                @endif
                @if (session('error'))
                    <div class="alert alert-danger alert-dismissible fade show" role="alert">
                        {{ session('error') }}
                        <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
                    </div>
                @endif

                <form action="{{ route('inward.update', $inward) }}" method="POST" enctype="multipart/form-data"
                    id="inward-form">
                    @csrf
                    @method('PUT')

                    <div class="mb-4">
                        <div class="card-header bg-light">
                            <h5>Inward & Vehicle Details</h5>
                        </div>
                        <div class="card-body row">
                            <div class="col-md-4 mb-3">
                                <label for="plant_id" class="form-label required">Plant</label>
                                <span class="text-danger">*</span>
                                <select name="plant_id" id="plant_id"
                                    class="form-select @error('plant_id') is-invalid @enderror" required
                                    onchange="handleChange(this)">
                                    <option value="">--Select Plant--</option>
                                    @foreach ($plants as $plant)
                                        <option value="{{ $plant->id }}"
                                            {{ old('plant_id', $inward->plant_id) == $plant->id ? 'selected' : '' }}>
                                            {{ $plant->name }}
                                        </option>
                                    @endforeach
                                </select>
                                @error('plant_id')
                                    <div class="invalid-feedback">{{ $message }}</div>
                                @else
                                    <div class="invalid-feedback">Please select a plant.</div>
                                @enderror
                            </div>
                            <div class="col-md-4 mb-3">
                                <label for="inward_code" class="form-label required">Inward Code</label>
                                <span class="text-danger">*</span>
                                <input type="text" name="inward_code" id="inward_code"
                                    class="form-control @error('inward_code') is-invalid @enderror"
                                    value="{{ old('inward_code', $inward->inward_code) }}" readonly required>
                                @error('inward_code')
                                    <div class="invalid-feedback">{{ $message }}</div>
                                @else
                                    <div class="invalid-feedback">Inward code is required.</div>
                                @enderror
                            </div>
                            <div class="col-md-4 mb-3">
                                <label for="inward_date" class="form-label required">Inward Date</label>
                                <span class="text-danger">*</span>
                                <input type="date" name="inward_date" id="inward_date"
                                    class="form-control @error('inward_date') is-invalid @enderror"
                                    value="{{ old('inward_date', $inward->inward_date) }}" required
                                    max="{{ now()->format('Y-m-d') }}">
                                @error('inward_date')
                                    <div class="invalid-feedback">{{ $message }}</div>
                                @else
                                    <div class="invalid-feedback">Please select a valid date.</div>
                                @enderror
                            </div>
                            <div class="col-md-4 mb-3">
                                <label for="grn_no" class="form-label required">GRN No.</label>
                                <span class="text-danger">*</span>
                                <input type="text" name="grn_no" id="grn_no"
                                    class="form-control @error('grn_no') is-invalid @enderror"
                                    value="{{ old('grn_no', $inward->grn_no) }}" readonly required>
                                @error('grn_no')
                                    <div class="invalid-feedback">{{ $message }}</div>
                                @else
                                    <div class="invalid-feedback">GRN No. is required.</div>
                                @enderror
                            </div>
                            <div class="col-md-4 mb-3">
                                <label for="vehicle_registration_number" class="form-label required">Vehicle Registration
                                    Number</label>
                                <span class="text-danger">*</span>
                                <input type="text" name="vehicle_registration_number" id="vehicle_registration_number"
                                    class="form-control @error('vehicle_registration_number') is-invalid @enderror"
                                    value="{{ old('vehicle_registration_number', $inward->vehicle_registration_number) }}"
                                    required maxlength="20">
                                @error('vehicle_registration_number')
                                    <div class="invalid-feedback">{{ $message }}</div>
                                @else
                                    <div class="invalid-feedback">Vehicle registration number is required.</div>
                                @enderror
                            </div>
                            <div class="col-md-4 mb-3">
                                <label for="challan_no" class="form-label required">57 F4 Challan No</label>
                                <span class="text-danger">*</span>
                                <input type="text" name="challan_no" id="challan_no"
                                    class="form-control @error('challan_no') is-invalid @enderror"
                                    value="{{ old('challan_no', $inward->challan_no) }}" required maxlength="50">
                                @error('challan_no')
                                    <div class="invalid-feedback">{{ $message }}</div>
                                @else
                                    <div class="invalid-feedback">Challan No. is required.</div>
                                @enderror
                            </div>
                            <div class="col-md-4 mb-3">
                                <label for="challan_date" class="form-label required">Challan Date</label>
                                <span class="text-danger">*</span>
                                <input type="date" name="challan_date" id="challan_date"
                                    class="form-control @error('challan_date') is-invalid @enderror"
                                    value="{{ old('challan_date', $inward->challan_date) }}" required
                                    max="{{ now()->format('Y-m-d') }}">
                                @error('challan_date')
                                    <div class="invalid-feedback">{{ $message }}</div>
                                @else
                                    <div class="invalid-feedback">Please select a valid challan date.</div>
                                @enderror
                            </div>
                        </div>
                    </div>

                    <div class="mb-4">
                        <div class="card-header bg-light">
                            <h5>Client Details</h5>
                        </div>
                        <div class="card-body row">
                            <div class="col-md-4 mb-3">
                                <label for="client_id" class="form-label required">Client Name</label>
                                <span class="text-danger">*</span>
                                <select name="client_id" id="client_id"
                                    class="form-select @error('client_id') is-invalid @enderror" required>
                                    <option value="">--Select Client Name--</option>
                                    @foreach ($clients as $client)
                                        <option value="{{ $client->client_id }}"
                                            {{ old('client_id', $inward->client_id) == $client->client_id ? 'selected' : '' }}>
                                            {{ $client->client_name }}
                                        </option>
                                    @endforeach
                                </select>
                                @error('client_id')
                                    <div class="invalid-feedback">{{ $message }}</div>
                                @else
                                    <div class="invalid-feedback">Please select a client.</div>
                                @enderror
                            </div>
                            <div class="col-md-4 mb-3">
                                <label for="manufacturer_id" class="form-label required">Manufacturer</label>
                                <span class="text-danger">*</span>
                                <select name="manufacturer_id" id="manufacturer_id"
                                    class="form-select @error('manufacturer_id') is-invalid @enderror" required>
                                    <option value="">--Select Manufacturer Name--</option>
                                    @foreach ($manufacturers as $manufacturer)
                                        <option value="{{ $manufacturer->manufacturer_id }}"
                                            {{ old('manufacturer_id', $inward->manufacturer_id) == $manufacturer->manufacturer_id ? 'selected' : '' }}>
                                            {{ $manufacturer->manufacturer_name }}
                                        </option>
                                    @endforeach
                                </select>
                                @error('manufacturer_id')
                                    <div class="invalid-feedback">{{ $message }}</div>
                                @else
                                    <div class="invalid-feedback">Please select a manufacturer.</div>
                                @enderror
                            </div>
                            <div class="col-md-4 mb-3">
                                <label for="invoice_no" class="form-label">Client Invoice Number</label>
                                <input type="text" name="invoice_no" id="invoice_no"
                                    class="form-control @error('invoice_no') is-invalid @enderror"
                                    value="{{ old('invoice_no', $inward->invoice_no) }}" maxlength="50">
                                @error('invoice_no')
                                    <div class="invalid-feedback">{{ $message }}</div>
                                @else
                                    <div class="invalid-feedback">Invalid invoice number.</div>
                                @enderror
                            </div>
                            <div class="col-md-4 mb-3">
                                <label for="invoice_date" class="form-label required">Invoice Date</label>
                                <span class="text-danger">*</span>
                                <input type="date" name="invoice_date" id="invoice_date"
                                    class="form-control @error('invoice_date') is-invalid @enderror"
                                    value="{{ old('invoice_date', $inward->invoice_date) }}" required
                                    max="{{ now()->format('Y-m-d') }}">
                                @error('invoice_date')
                                    <div class="invalid-feedback">{{ $message }}</div>
                                @else
                                    <div class="invalid-feedback">Please select a valid invoice date.</div>
                                @enderror
                            </div>
                            <div class="col-md-4 mb-3">
                                <label for="upload_invoice" class="form-label">Upload Invoice</label>
                                <input type="file" name="upload_invoice" id="upload_invoice"
                                    class="form-control @error('upload_invoice') is-invalid @enderror"
                                    accept=".pdf,.jpg,.png">
                                @if ($inward->upload_invoice)
                                    <small class="form-text text-muted">Current file: <a
                                            href="{{ Storage::url($inward->upload_invoice) }}"
                                            target="_blank">View</a></small>
                                @endif
                                @error('upload_invoice')
                                    <div class="invalid-feedback">{{ $message }}</div>
                                @else
                                    <div class="invalid-feedback">Please upload a valid invoice file (PDF, JPG, PNG).</div>
                                @enderror
                            </div>
                            <div class="col-md-4 mb-3">
                                <label for="hsn_code" class="form-label required">HSN Code</label>
                                <span class="text-danger">*</span>
                                <input type="text" name="hsn_code" id="hsn_code"
                                    class="form-control @error('hsn_code') is-invalid @enderror"
                                    value="{{ old('hsn_code', $inward->hsn_code) }}" required maxlength="50">
                                @error('hsn_code')
                                    <div class="invalid-feedback">{{ $message }}</div>
                                @else
                                    <div class="invalid-feedback">HSN code is required.</div>
                                @enderror
                            </div>
                        </div>
                    </div>

                    <div class="mb-4">
                        <div class="card-header bg-light d-flex justify-content-between align-items-center">
                            <h5>Coil Details</h5>
                            <button type="button" class="btn btn-primary btn-sm" id="add-coil">Add Coil</button>
                        </div>
                        <div class="card-body" id="coil-container">
                            @if ($inward->coils->isNotEmpty())
                                @foreach ($inward->coils as $index => $coil)
                                    <div class="coil-row" id="coil-{{ $index }}"
                                        data-index="{{ $index }}">
                                        <input type="hidden" name="coils[{{ $index }}][id]"
                                            value="{{ $coil->id }}">
                                        <div class="d-flex justify-content-between align-items-center mb-2">
                                            <h6>Coil {{ $index + 1 }}</h6>
                                            <span class="remove-coil">×</span>
                                        </div>
                                        <div class="row">
                                            <div class="col-md-4 mb-3">
                                                <label for="coils[{{ $index }}][crr_no]"
                                                    class="form-label required">CRR
                                                    No.</label>
                                                <span class="text-danger">*</span>
                                                <input type="text" name="coils[{{ $index }}][crr_no]"
                                                    id="coils[{{ $index }}][crr_no]"
                                                    class="form-control crr-no @error('coils.' . $index . '.crr_no') is-invalid @enderror"
                                                    value="{{ old('coils.' . $index . '.crr_no', $coil->crr_no) }}"
                                                    readonly required>
                                                @error('coils.' . $index . '.crr_no')
                                                    <div class="invalid-feedback">{{ $message }}</div>
                                                @else
                                                    <div class="invalid-feedback">CRR No. is required.</div>
                                                @enderror
                                            </div>
                                            <div class="col-md-4 mb-3">
                                                <label for="coils[{{ $index }}][coil_no]"
                                                    class="form-label required">Customer
                                                    Coil No./Batch No.</label>
                                                <span class="text-danger">*</span>
                                                <input type="text" name="coils[{{ $index }}][coil_no]"
                                                    id="coils[{{ $index }}][coil_no]"
                                                    class="form-control coil-no @error('coils.' . $index . '.coil_no') is-invalid @enderror"
                                                    value="{{ old('coils.' . $index . '.coil_no', $coil->coil_no) }}"
                                                    required maxlength="50">
                                                @error('coils.' . $index . '.coil_no')
                                                    <div class="invalid-feedback">{{ $message }}</div>
                                                @else
                                                    <div class="invalid-feedback">Coil No. is required.</div>
                                                @enderror
                                            </div>
                                            <div class="col-md-4 mb-3">
                                                <label for="grade_id_{{ $index }}"
                                                    class="form-label required">Grade</label>
                                                <span class="text-danger">*</span>
                                                <input type="text"
                                                    class="form-control grade-input @error('coils.' . $index . '.grade_id') is-invalid @enderror"
                                                    id="grade_id_{{ $index }}" data-index="{{ $index }}"
                                                    data-selected="{{ $coil->grade_id ? 'true' : 'false' }}"
                                                    value="{{ old('coils.' . $index . '.grade_name', $coil->grade->grade_name ?? '') }}"
                                                    required>
                                                <input type="hidden" name="coils[{{ $index }}][grade_id]"
                                                    id="grade_id_hidden_{{ $index }}" class="grade-id-hidden"
                                                    value="{{ old('coils.' . $index . '.grade_id', $coil->grade_id ?? '') }}">
                                                <div id="grade_suggestions_{{ $index }}"
                                                    class="autocomplete-suggestions"></div>
                                                @error('coils.' . $index . '.grade_id')
                                                    <div class="invalid-feedback">{{ $message }}</div>
                                                @else
                                                    <div class="invalid-feedback">Please select a grade.</div>
                                                @enderror
                                            </div>
                                            <div class="col-md-4 mb-3">
                                                <label for="coils[{{ $index }}][thickness]"
                                                    class="form-label required">Thickness
                                                    (in mm)</label>
                                                <span class="text-danger">*</span>
                                                <input type="number" step="0.01"
                                                    name="coils[{{ $index }}][thickness]"
                                                    id="coils[{{ $index }}][thickness]"
                                                    class="form-control @error('coils.' . $index . '.thickness') is-invalid @enderror"
                                                    value="{{ old('coils.' . $index . '.thickness', $coil->thickness) }}"
                                                    required min="0">
                                                @error('coils.' . $index . '.thickness')
                                                    <div class="invalid-feedback">{{ $message }}</div>
                                                @else
                                                    <div class="invalid-feedback">Thickness is required and must be positive.</div>
                                                @enderror
                                            </div>
                                            <div class="col-md-4 mb-3">
                                                <label for="coils[{{ $index }}][width]"
                                                    class="form-label required">Width (in
                                                    mm)</label>
                                                <span class="text-danger">*</span>
                                                <input type="number" step="0.01"
                                                    name="coils[{{ $index }}][width]"
                                                    id="coils[{{ $index }}][width]"
                                                    class="form-control @error('coils.' . $index . '.width') is-invalid @enderror"
                                                    value="{{ old('coils.' . $index . '.width', $coil->width) }}" required
                                                    min="0">
                                                @error('coils.' . $index . '.width')
                                                    <div class="invalid-feedback">{{ $message }}</div>
                                                @else
                                                    <div class="invalid-feedback">Width is required and must be positive.</div>
                                                @enderror
                                            </div>
                                            <div class="col-md-4 mb-3">
                                                <label for="coils[{{ $index }}][net_weight]"
                                                    class="form-label required">Net
                                                    Weight (in Ton)</label>
                                                <span class="text-danger">*</span>
                                                <input type="number" step="0.001"
                                                    name="coils[{{ $index }}][net_weight]"
                                                    id="coils[{{ $index }}][net_weight]"
                                                    class="form-control @error('coils.' . $index . '.net_weight') is-invalid @enderror"
                                                    value="{{ old('coils.' . $index . '.net_weight', $coil->net_weight) }}"
                                                    required min="0">
                                                @error('coils.' . $index . '.net_weight')
                                                    <div class="invalid-feedback">{{ $message }}</div>
                                                @else
                                                    <div class="invalid-feedback">Net weight is required and must be positive.</div>
                                                @enderror
                                            </div>
                                            <div class="col-md-4 mb-3">
                                                <label for="coils[{{ $index }}][gross_weight]"
                                                    class="form-label required">Gross
                                                    Weight (in Ton)</label>
                                                <span class="text-danger">*</span>
                                                <input type="number" step="0.001"
                                                    name="coils[{{ $index }}][gross_weight]"
                                                    id="coils[{{ $index }}][gross_weight]"
                                                    class="form-control @error('coils.' . $index . '.gross_weight') is-invalid @enderror"
                                                    value="{{ old('coils.' . $index . '.gross_weight', $coil->gross_weight) }}"
                                                    required min="0">
                                                @error('coils.' . $index . '.gross_weight')
                                                    <div class="invalid-feedback">{{ $message }}</div>
                                                @else
                                                    <div class="invalid-feedback">Gross weight is required and must be positive.</div>
                                                @enderror
                                            </div>
                                            <div style="display: none;">
                                            <div class="col-md-4 mb-3">
                                                <label for="coils[{{ $index }}][used_coil]"
                                                    class="form-label required">Used Coil (in Ton)</label>
                                                <span class="text-danger">*</span>
                                                <input type="number" step="0.001"
                                                    name="coils[{{ $index }}][used_coil]"
                                                    id="coils[{{ $index }}][used_coil]"
                                                    class="form-control @error('coils.' . $index . '.used_coil') is-invalid @enderror"
                                                    value="{{ old('coils.' . $index . '.used_coil', $coil->used_coil ?? 0) }}"
                                                    required min="0">
                                                @error('coils.' . $index . '.used_coil')
                                                    <div class="invalid-feedback">{{ $message }}</div>
                                                @else
                                                    <div class="invalid-feedback">Used coil weight is required and must be positive.</div>
                                                @enderror
                                            </div>
                                            <div class="col-md-4 mb-3">
                                                <label for="coils[{{ $index }}][balance_coil]"
                                                    class="form-label required">Balance Coil (in Ton)</label>
                                                <span class="text-danger">*</span>
                                                <input type="number" step="0.001"
                                                    name="coils[{{ $index }}][balance_coil]"
                                                    id="coils[{{ $index }}][balance_coil]"
                                                    class="form-control @error('coils.' . $index . '.balance_coil') is-invalid @enderror"
                                                    value="{{ old('coils.' . $index . '.balance_coil', $coil->balance_coil ?? $coil->net_weight) }}"
                                                    required min="0">
                                                @error('coils.' . $index . '.balance_coil')
                                                    <div class="invalid-feedback">{{ $message }}</div>
                                                @else
                                                    <div class="invalid-feedback">Balance coil weight is required and must be positive.</div>
                                                @enderror
                                            </div>
                                            <div class="col-md-4 mb-3">
                                                <label for="coils[{{ $index }}][scrap_wt]"
                                                    class="form-label required">Scrap Weight (in Ton)</label>
                                                <span class="text-danger">*</span>
                                                <input type="number" step="0.001"
                                                    name="coils[{{ $index }}][scrap_wt]"
                                                    id="coils[{{ $index }}][scrap_wt]"
                                                    class="form-control @error('coils.' . $index . '.scrap_wt') is-invalid @enderror"
                                                    value="{{ old('coils.' . $index . '.scrap_wt', $coil->scrap_wt ?? 0) }}"
                                                    required min="0">
                                                @error('coils.' . $index . '.scrap_wt')
                                                    <div class="invalid-feedback">{{ $message }}</div>
                                                @else
                                                    <div class="invalid-feedback">Scrap weight is required and must be positive.</div>
                                                @enderror
                                            </div>
                                            </div>
                                            <div class="col-md-4 mb-3">
                                                <label for="coils[{{ $index }}][material_type_id]"
                                                    class="form-label required">Material
                                                    Type</label>
                                                <span class="text-danger">*</span>
                                                <select class="form-control material-type-select @error('coils.' . $index . '.material_type_id') is-invalid @enderror"
                                                    id="coils[{{ $index }}][material_type_id]"
                                                    name="coils[{{ $index }}][material_type_id]" required>
                                                    <option value="">--Select Material Type--</option>
                                                    @foreach ($materialTypes as $materialType)
                                                        <option value="{{ $materialType->id }}"
                                                            data-density="{{ $materialType->density }}"
                                                            {{ old('coils.' . $index . '.material_type_id', $coil->material_type_id) == $materialType->id ? 'selected' : '' }}>
                                                            {{ $materialType->material_name }}
                                                        </option>
                                                    @endforeach
                                                </select>
                                                @error('coils.' . $index . '.material_type_id')
                                                    <div class="invalid-feedback">{{ $message }}</div>
                                                @else
                                                    <div class="invalid-feedback">Material type is required.</div>
                                                @enderror
                                            </div>
                                            <div class="col-md-4 mb-3">
                                                <label for="coils[{{ $index }}][basic_rate_per_ton]"
                                                    class="form-label required">Basic
                                                    Rate Per Ton</label>
                                                <span class="text-danger">*</span>
                                                <input type="number" step="0.01"
                                                    name="coils[{{ $index }}][basic_rate_per_ton]"
                                                    id="coils[{{ $index }}][basic_rate_per_ton]"
                                                    class="form-control @error('coils.' . $index . '.basic_rate_per_ton') is-invalid @enderror"
                                                    value="{{ old('coils.' . $index . '.basic_rate_per_ton', $coil->basic_rate_per_ton) }}"
                                                    required min="0">
                                                @error('coils.' . $index . '.basic_rate_per_ton')
                                                    <div class="invalid-feedback">{{ $message }}</div>
                                                @else
                                                    <div class="invalid-feedback">Basic rate is required and must be positive.</div>
                                                @enderror
                                            </div>
                                            <div class="col-md-4 mb-3">
                                                <label for="coils[{{ $index }}][location]"
                                                    class="form-label">Location</label>
                                                <input type="text" name="coils[{{ $index }}][location]"
                                                    id="coils[{{ $index }}][location]"
                                                    class="form-control @error('coils.' . $index . '.location') is-invalid @enderror"
                                                    value="{{ old('coils.' . $index . '.location', $coil->location) }}"
                                                    maxlength="255">
                                                @error('coils.' . $index . '.location')
                                                    <div class="invalid-feedback">{{ $message }}</div>
                                                @else
                                                    <div class="invalid-feedback">Invalid location.</div>
                                                @enderror
                                            </div>
                                            <div class="col-md-4 mb-3">
                                                <label for="coils[{{ $index }}][remark]"
                                                    class="form-label">Remark</label>
                                                <input type="text" name="coils[{{ $index }}][remark]"
                                                    id="coils[{{ $index }}][remark]"
                                                    class="form-control @error('coils.' . $index . '.remark') is-invalid @enderror"
                                                    value="{{ old('coils.' . $index . '.remark', $coil->remark) }}"
                                                    maxlength="255">
                                                @error('coils.' . $index . '.remark')
                                                    <div class="invalid-feedback">{{ $message }}</div>
                                                @else
                                                    <div class="invalid-feedback">Invalid remark.</div>
                                                @enderror
                                            </div>
                                        </div>
                                    </div>
                                @endforeach
                            @endif

                            <div class="coil-row coil-template" id="coil-template" data-index="template">
                                <div class="d-flex justify-content-between align-items-center mb-2">
                                    <h6>Coil 1</h6>
                                    <span class="remove-coil">×</span>
                                </div>
                                <div class="row">
                                    <div class="col-md-4 mb-3">
                                        <label for="coils[template][crr_no]" class="form-label required">CRR No.</label>
                                        <span class="text-danger">*</span>
                                        <input type="text" name="coils[template][crr_no]" id="coils[template][crr_no]"
                                            class="form-control crr-no" readonly disabled>
                                        <div class="invalid-feedback">CRR No. is required.</div>
                                    </div>
                                    <div class="col-md-4 mb-3">
                                        <label for="coils[template][coil_no]" class="form-label required">Customer Coil
                                            No./Batch No.</label>
                                        <span class="text-danger">*</span>
                                        <input type="text" name="coils[template][coil_no]"
                                            id="coils[template][coil_no]" class="form-control coil-no"
                                            maxlength="50" disabled>
                                        <div class="invalid-feedback">Coil No. is required.</div>
                                    </div>
                                    <div class="col-md-4 mb-3">
                                        <label for="grade_id_template" class="form-label required">Grade</label>
                                        <span class="text-danger">*</span>
                                        <input type="text" class="form-control grade-input" id="grade_id_template"
                                            data-index="template" data-selected="false" disabled>
                                        <input type="hidden" name="coils[template][grade_id]"
                                            id="grade_id_hidden_template" class="grade-id-hidden" disabled>
                                        <div id="grade_suggestions_template" class="autocomplete-suggestions"></div>
                                        <div class="invalid-feedback">Please select a grade.</div>
                                    </div>
                                    <div class="col-md-4 mb-3">
                                        <label for="coils[template][thickness]" class="form-label required">Thickness (in
                                            mm)</label>
                                        <span class="text-danger">*</span>
                                        <input type="number" step="0.01" name="coils[template][thickness]"
                                            id="coils[template][thickness]" class="form-control" min="0" disabled>
                                        <div class="invalid-feedback">Thickness is required and must be positive.</div>
                                    </div>
                                    <div class="col-md-4 mb-3">
                                        <label for="coils[template][width]" class="form-label required">Width (in
                                            mm)</label>
                                        <span class="text-danger">*</span>
                                        <input type="number" step="0.01" name="coils[template][width]"
                                            id="coils[template][width]" class="form-control" min="0" disabled>
                                        <div class="invalid-feedback">Width is required and must be positive.</div>
                                    </div>
                                    <div class="col-md-4 mb-3">
                                        <label for="coils[template][net_weight]" class="form-label required">Net Weight
                                            (in Ton)</label>
                                        <span class="text-danger">*</span>
                                        <input type="number" step="0.001" name="coils[template][net_weight]"
                                            id="coils[template][net_weight]" class="form-control" min="0" disabled>
                                        <div class="invalid-feedback">Net weight is required and must be positive.</div>
                                    </div>
                                    <div class="col-md-4 mb-3">
                                        <label for="coils[template][gross_weight]" class="form-label required">Gross
                                            Weight (in Ton)</label>
                                        <span class="text-danger">*</span>
                                        <input type="number" step="0.001" name="coils[template][gross_weight]"
                                            id="coils[template][gross_weight]" class="form-control" min="0" disabled>
                                        <div class="invalid-feedback">Gross weight is required and must be positive.</div>
                                    </div>
                                    <div class="col-md-4 mb-3">
                                        <label for="coils[template][used_coil]" class="form-label required">Used Coil (in Ton)</label>
                                        <span class="text-danger">*</span>
                                        <input type="number" step="0.001" name="coils[template][used_coil]"
                                            id="coils[template][used_coil]" class="form-control" min="0" disabled>
                                        <div class="invalid-feedback">Used coil weight is required and must be positive.</div>
                                    </div>
                                    <div class="col-md-4 mb-3">
                                        <label for="coils[template][balance_coil]" class="form-label required">Balance Coil (in Ton)</label>
                                        <span class="text-danger">*</span>
                                        <input type="number" step="0.001" name="coils[template][balance_coil]"
                                            id="coils[template][balance_coil]" class="form-control" min="0" disabled>
                                        <div class="invalid-feedback">Balance coil weight is required and must be positive.</div>
                                    </div>
                                    <div class="col-md-4 mb-3">
                                        <label for="coils[template][scrap_wt]" class="form-label required">Scrap Weight (in Ton)</label>
                                        <span class="text-danger">*</span>
                                        <input type="number" step="0.001" name="coils[template][scrap_wt]"
                                            id="coils[template][scrap_wt]" class="form-control" min="0" disabled>
                                        <div class="invalid-feedback">Scrap weight is required and must be positive.</div>
                                    </div>
                                    <div class="col-md-4 mb-3">
                                        <label for="coils[template][material_type_id]"
                                            class="form-label required">Material Type</label>
                                        <span class="text-danger">*</span>
                                        <select class="form-control material-type-select"
                                            id="coils[template][material_type_id]" name="coils[template][material_type_id]"
                                            disabled>
                                            <option value="">--Select Material Type--</option>
                                            @foreach ($materialTypes as $materialType)
                                                <option value="{{ $materialType->id }}"
                                                    data-density="{{ $materialType->density }}">
                                                    {{ $materialType->material_name }}
                                                </option>
                                            @endforeach
                                        </select>
                                        <div class="invalid-feedback">Material type is required.</div>
                                    </div>
                                    <div class="col-md-4 mb-3">
                                        <label for="coils[template][basic_rate_per_ton]"
                                            class="form-label required">Basic Rate Per Ton</label>
                                        <span class="text-danger">*</span>
                                        <input type="number" step="0.01" name="coils[template][basic_rate_per_ton]"
                                            id="coils[template][basic_rate_per_ton]" class="form-control" min="0" disabled>
                                        <div class="invalid-feedback">Basic rate is required and must be positive.</div>
                                    </div>
                                    <div class="col-md-4 mb-3">
                                        <label for="coils[template][location]" class="form-label">Location</label>
                                        <input type="text" name="coils[template][location]"
                                            id="coils[template][location]" class="form-control" maxlength="255" disabled>
                                        <div class="invalid-feedback">Invalid location.</div>
                                    </div>
                                    <div class="col-md-4 mb-3">
                                        <label for="coils[template][remark]" class="form-label">Remark</label>
                                        <input type="text" name="coils[template][remark]" id="coils[template][remark]"
                                            class="form-control" maxlength="255" disabled>
                                        <div class="invalid-feedback">Invalid remark.</div>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>

                    <div class="col-md-12 d-flex justify-content-end">
                        <a href="{{ route('inward.index') }}" class="btn btn-danger btn-sm px-4 me-2">Cancel</a>
                        <button type="submit" class="btn btn-success btn-sm px-4">Update Inward</button>
                    </div>
                </form>
            </div>
        </div>
    </div>
@endsection

@section('scripts')
    <script src="https://code.jquery.com/jquery-3.6.0.min.js"></script>
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        jQuery.noConflict();
        jQuery(document).ready(function($) {
            console.log('jQuery loaded, script running');

            $('#vehicle_registration_number, #challan_no, input[name*="coil_no"]').on('input', function() {
                $(this).val($(this).val().toUpperCase());
            });

            const today = new Date().toISOString().split('T')[0];
            $('#inward_date, #invoice_date, #challan_date').attr('max', today);

            let crrState = {
                letterCode: 'AA',
                number: 1
            };
            const lastCrrNo = $('.crr-no').last().val() || 'CRRAA00001';
            const crrMatch = lastCrrNo.match(/CRR([A-Z]{2})(\d{5})/);
            if (crrMatch) {
                crrState.letterCode = crrMatch[1];
                crrState.number = parseInt(crrMatch[2]) + 1;
            }
            console.log('Initial CRR State:', crrState);

            function generateCrrNo() {
                const prefix = 'CRR';
                const letterCode = crrState.letterCode;
                const number = crrState.number.toString().padStart(5, '0');
                const crrNo = `${prefix}${letterCode}${number}`;
                console.log('Generated CRR No.:', crrNo);
                crrState.number++;
                if (crrState.number > 99999) {
                    crrState.number = 1;
                    crrState.letterCode = incrementLetterCode(crrState.letterCode);
                }
                return crrNo;
            }

            function incrementLetterCode(code) {
                let [first, second] = code.split('');
                if (second === 'Z') {
                    if (first === 'Z') {
                        return 'AA';
                    }
                    first = String.fromCharCode(first.charCodeAt(0) + 1);
                    second = 'A';
                } else {
                    second = String.fromCharCode(second.charCodeAt(0) + 1);
                }
                return `${first}${second}`;
            }

            $(document).on('input', '.grade-input', function() {
                const $this = $(this);
                const query = $this.val().trim().toLowerCase();
                const index = $this.data('index');
                const suggestionsContainer = `#grade_suggestions_${index}`;
                console.log(`Grade input for index ${index}:`, query);

                $this.data('selected', false);
                $(`#grade_id_hidden_${index}`).val('');

                if (query.length > 0) {
                    $.ajax({
                        url: '{{ route('grades.search') }}',
                        method: 'GET',
                        data: { query: query },
                        success: function(data) {
                            let suggestions = '';
                            if (data.length > 0) {
                                suggestions = '<ul class="list-unstyled border p-2 bg-white">';
                                data.forEach(grade => {
                                    suggestions +=
                                        `<li class="suggestion-item" data-index="${index}" data-grade-id="${grade.id}">${grade.name}</li>`;
                                });
                                suggestions += '</ul>';
                            }
                            $(suggestionsContainer).html(suggestions).show();
                        },
                        error: function() {
                            console.error('Error fetching grades');
                        }
                    });
                } else {
                    $(suggestionsContainer).html('').hide();
                }
            });

            $(document).on('click', '.suggestion-item', function() {
                const index = $(this).data('index');
                const gradeId = $(this).data('grade-id');
                const gradeName = $(this).text();
                console.log(`Selected grade for index ${index}:`, {
                    id: gradeId,
                    name: gradeName
                });

                $(`#grade_id_${index}`)
                    .val(gradeName)
                    .data('selected', true)
                    .removeClass('is-invalid');
                $(`#grade_id_hidden_${index}`).val(gradeId);
                $(`#grade_suggestions_${index}`).html('').hide();
            });

            $(document).on('click', function(e) {
                if (!$(e.target).closest('.grade-input, .autocomplete-suggestions').length) {
                    $('.autocomplete-suggestions').html('').hide();
                }
            });

            $(document).on('blur', '.coil-no', function() {
                const coilNo = $(this).val().toUpperCase().trim();
                if (coilNo) {
                    $.ajax({
                        url: '{{ route('checkCoilNo') }}',
                        method: 'POST',
                        data: {
                            coil_no: coilNo,
                            _token: '{{ csrf_token() }}',
                            except_id: {{ $inward->id }}
                        },
                        success: function(response) {
                            if (response.exists) {
                                alert('Coil No. already exists!');
                                $(this).val('').addClass('is-invalid');
                            }
                        }.bind(this),
                        error: function() {
                            alert('Error checking Coil No.');
                        }
                    });
                }
            });

            let coilIndex = {{ $inward->coils->count() }};
            $('#add-coil').on('click', function(e) {
                e.preventDefault();
                console.log('Add Coil button clicked, new index:', coilIndex);

                try {
                    let $coilRow;
                    if ($('.coil-row:not(.coil-template)').length > 0) {
                        $coilRow = $('.coil-row:not(.coil-template):first').clone(true);
                    } else {
                        $coilRow = $('#coil-template').clone(true);
                        $coilRow.removeClass('coil-template').css('display', 'block');
                    }

                    $coilRow.attr('id', `coil-${coilIndex}`);
                    $coilRow.attr('data-index', coilIndex);
                    $coilRow.find('h6').text(`Coil ${coilIndex + 1}`);
                    $coilRow.addClass('new');
                    setTimeout(() => $coilRow.removeClass('new'), 1000);

                    const newCrrNo = generateCrrNo();

                    $coilRow.find('.grade-input')
                        .attr('id', `grade_id_${coilIndex}`)
                        .attr('data-index', coilIndex)
                        .data('index', coilIndex)
                        .data('selected', false)
                        .val('')
                        .prop('required', true)
                        .prop('disabled', false);

                    $coilRow.find('.grade-id-hidden')
                        .attr('id', `grade_id_hidden_${coilIndex}`)
                        .attr('name', `coils[${coilIndex}][grade_id]`)
                        .val('')
                        .prop('disabled', false);

                    $coilRow.find('.autocomplete-suggestions')
                        .attr('id', `grade_suggestions_${coilIndex}`)
                        .html('')
                        .hide();

                    $coilRow.find('input, select').each(function() {
                        const $this = $(this);
                        const oldName = $this.attr('name');
                        const oldId = $this.attr('id');

                        if (oldName && oldId) {
                            const newName = oldName.replace(/coils\[(\w+)\]/,
                                `coils[${coilIndex}]`);
                            const newId = oldId.replace(/coils\[(\w+)\]/, `coils[${coilIndex}]`);

                            $this
                                .attr('name', newName)
                                .attr('id', newId)
                                .removeClass('is-invalid')
                                .prop('disabled', false);

                            if ($this.hasClass('crr-no')) {
                                $this.val(newCrrNo).prop('readonly', true).prop('required', true);
                            } else if ($this.hasClass('coil-no') || $this.hasClass('grade-input') ||
                                       $this.attr('name').includes('[thickness]') ||
                                       $this.attr('name').includes('[width]') ||
                                       $this.attr('name').includes('[net_weight]') ||
                                       $this.attr('name').includes('[gross_weight]') ||
                                       $this.attr('name').includes('[used_coil]') ||
                                       $this.attr('name').includes('[balance_coil]') ||
                                       $this.attr('name').includes('[scrap_wt]') ||
                                       $this.attr('name').includes('[basic_rate_per_ton]')) {
                                $this.val('').prop('required', true);
                            } else if ($this.hasClass('material-type-select')) {
                                $this.val('').prop('required', true);
                            } else {
                                $this.val('');
                            }
                        }
                    });

                    $coilRow.find('.invalid-feedback').hide();

                    $('#coil-container').append($coilRow);
                    coilIndex++;
                    updateRemoveButtons();
                    console.log('New coil row added successfully');
                } catch (error) {
                    console.error('Error adding coil row:', error);
                    alert('Failed to add new coil. Please check the console for details.');
                }
            });

            $(document).on('click', '.remove-coil', function(e) {
                e.preventDefault();
                if ($('.coil-row:not(.coil-template)').length > 1) {
                    $(this).closest('.coil-row').remove();
                    updateCoilHeaders();
                    updateRemoveButtons();
                }
            });

            function updateCoilHeaders() {
                $('.coil-row:not(.coil-template)').each(function(index) {
                    $(this).find('h6').text(`Coil ${index + 1}`);
                });
            }

            function updateRemoveButtons() {
                $('.remove-coil').show();
                if ($('.coil-row:not(.coil-template)').length <= 1) {
                    $('.coil-row:not(.coil-template)').find('.remove-coil').hide();
                }
            }

            $(document).on('input', 'input[name*="coil_no"]', function() {
                const $input = $(this);
                const value = $input.val().trim();
                const field = 'Coil No.';
                if (value) {
                    const values = $('input[name*="coil_no"]').map(function() {
                        return $(this).val().trim().toUpperCase();
                    }).get();
                    const duplicates = values.filter((item, index) => item === value.toUpperCase() && values
                        .indexOf(item) !== index);
                    if (duplicates.length > 0) {
                        $input.addClass('is-invalid');
                        $input.siblings('.invalid-feedback').text(`${field} is duplicated in this form.`);
                    } else {
                        $input.removeClass('is-invalid');
                        $input.siblings('.invalid-feedback').text(`${field} is required.`);
                    }
                } else {
                    $input.addClass('is-invalid');
                    $input.siblings('.invalid-feedback').text(`${field} is required.`);
                }
            });

            $('#inward-form').on('submit', function(e) {
                let isValid = true;
                $('.grade-input').each(function() {
                    const $this = $(this);
                    if (!$this.data('selected') || !$this.val().trim()) {
                        $this.addClass('is-invalid');
                        isValid = false;
                    }
                });
                if (!isValid) {
                    e.preventDefault();
                    alert('Please select a valid grade for all coils.');
                }
            });

            updateRemoveButtons();

            @if ($errors->any())
                $('.alert-danger').removeClass('d-none');
            @endif

            function handleChange(select) {
                const selectedPlantId = select.value;
                console.log('Selected Plant ID:', selectedPlantId);

                const url = '{{ route('getplantclients', [':selectedPlantId']) }}'.replace(':selectedPlantId',
                    selectedPlantId);

                $.ajax({
                    url: url,
                    type: 'GET',
                    success: function(response) {
                        console.log('Clients response:', response);
                        if (response.status === 'success') {
                            const clients = response.clients;
                            $('select[name="client_id"]').empty().append(
                                '<option value="">--Select Client Name--</option>');
                            clients.forEach(client => {
                                $('select[name="client_id"]').append(
                                    `<option value="${client.client_id}">${client.client_name}</option>`
                                );
                            });
                        } else {
                            alert('No clients found for the selected plant.');
                        }
                    },
                    error: function(xhr) {
                        console.error('Error fetching clients:', xhr.responseText);
                        $('.alert-danger').removeClass('d-none').html(
                            '<strong>Error:</strong> Failed to fetch clients for the selected plant.'
                        );
                    }
                });
            }
        });
    </script>
@endsection
