@extends('layouts.structure')

@section('content')
    <style>
        .section-title {
            background-color: #f8f9fa;
            padding: 10px;
            margin-top: 20px;
            border-left: 5px solid #0d6efd;
        }

        .coil-details {
            background-color: #e9ecef;
            padding: 15px;
            border-radius: 5px;
        }

        .process-section {
            border: 1px solid #dee2e6;
            border-radius: 5px;
            padding: 15px;
            margin-bottom: 20px;
        }

        .balance-weight {
            font-weight: bold;
            color: #0d6efd;
        }

        .table th {
            background-color: #f8f9fa;
        }

        .dimension-select {
            background-color: #e9f7ef;
        }

        .process-buttons {
            display: flex;
            gap: 10px;
            margin-bottom: 20px;
        }

        .error-message {
            color: red;
            font-size: 0.9em;
            display: none;
        }

        @media print {
            button,
            .btn,
            .btn-close,
            .modal,
            .modal-backdrop,
            .process-buttons {
                display: none !important;
            }

            input.form-control,
            select.form-select {
                border: none !important;
                background: transparent !important;
                padding: 0 !important;
                font-size: 1rem !important;
                line-height: normal !important;
                -webkit-appearance: none !important;
                -moz-appearance: none !important;
                appearance: none !important;
            }

            .progress-bar {
                color: black !important;
                font-size: 12px !important;
                border: 1px solid #000 !important;
            }

            body {
                margin: 0 !important;
            }

            .container {
                padding: 10px !important;
            }

            .error-message,
            .section-title,
            .coil-details,
            .dimension-select {
                display: none !important;
                background-color: transparent !important;
                border: none !important;
            }

            .table {
                page-break-inside: auto !important;
            }

            tr {
                page-break-inside: avoid !important;
                page-break-after: auto !important;
            }
        }
    </style>

    <div class="container mt-4 mb-5">
        <h2 class="text-center mb-4">Edit Plan</h2>

        <!-- Debug: Display if no processes exist -->
        @if ($plan->processes->isEmpty())
            <div class="alert alert-warning">No processes found for this plan.</div>
        @else
            <p>Found {{ $plan->processes->count() }} processes.</p>
        @endif

        @if ($warning)
            <div class="alert alert-warning">{{ $warning }}</div>
        @endif

        <!-- Coil Details Section -->
        <div class="section-title">
            <h4>Coil Details</h4>
        </div>
        <div class="row mb-4 coil-details">
            <div class="col-md-4">
                <div class="mb-3">
                    <label for="client_id" class="form-label">Client Name</label>
                    <select class="form-control" id="client_id" name="client_id" required>
                        <option value="">Select</option>
                        @foreach ($clients as $clientall)
                            <option value="{{ $clientall->client_id }}"
                                {{ $plan->client_id == $clientall->client_id ? 'selected' : '' }}>
                                {{ $clientall->client_name }}</option>
                        @endforeach
                    </select>
                    <div id="clientNameError" class="error-message">Please select a client</div>
                    @if (!$plan->client)
                        <div class="text-danger mt-2">No client associated with this plan.</div>
                    @endif
                </div>
            </div>
            <div class="col-md-4">
                <div class="mb-3">
                    <label for="crr_no" class="form-label">CRR No.</label>
                    <select class="form-control" id="crr_no" name="coil_id" required>
                        <option value="">Select CRR No.</option>
                        @if ($plan->coil)
                            <option value="{{ $plan->coil_id }}" selected>
                                {{ $plan->coil->crr_no ?? 'N/A' }}
                            </option>
                        @endif
                    </select>
                    <div id="crrNoError" class="error-message">Please select a CRR No.</div>
                    @if (!$plan->coil)
                        <div class="text-danger mt-2">No coil associated with this plan.</div>
                    @endif
                </div>
            </div>
            <div class="col-md-4">
                <div class="mb-3">
                    <label for="grade_id" class="form-label">Grade</label>
                    <select class="form-control" name="grade_id" id="grade_id" style="background-color: white;">
                        <option value="">N/A</option>
                        @foreach ($grades as $grade)
                            <option value="{{ $grade->id }}" {{ $plan->grade_id == $grade->id ? 'selected' : '' }}>
                                {{ $grade->grade_name }}</option>
                        @endforeach
                    </select>
                    <div id="gradeError" class="error-message">Grade is required</div>
                </div>
            </div>
            <div class="col-md-3">
                <div class="mb-3">
                    <label for="width" class="form-label">Width (mm)</label>
                    <input type="number" class="form-control" id="width" value="{{ $plan->coil->width ?? '' }}"
                        readonly>
                    <div id="widthError" class="error-message">Width must be a positive number</div>
                </div>
            </div>
            <div class="col-md-3">
                <div class="mb-3">
                    <label for="thickness" class="form-label">Thickness (mm)</label>
                    <input type="number" class="form-control" id="thickness" value="{{ $plan->coil->thickness ?? '' }}"
                        readonly>
                    <div id="thicknessError" class="error-message">Thickness must be a positive number</div>
                </div>
            </div>
            <div class="col-md-3">
                <div class="mb-3">
                    <label for="density" class="form-label">Density</label>
                    <input type="number" class="form-control" id="density"
                        value="{{ $plan->coil->material_type_density ?? '' }}">
                    <input type="hidden" name="inward_id" value="{{ $plan->inward_id }}" id="inward_id" hidden>
                    <div id="densityError" class="error-message">Density must be a positive number</div>
                </div>
            </div>
            <div class="col-md-3">
                <div class="mb-3">
                    <label for="netWeight" class="form-label">Net Weight (TON)</label>
                    <input type="number" class="form-control" id="netWeight" value="{{ $plan->coil->net_weight ?? '' }}"
                        readonly>
                    <div id="netWeightError" class="error-message">Net Weight must be a positive number</div>
                </div>
            </div>
        </div>

        <!-- Summary Cards -->
        <div class="row mb-4">
            <div class="col-md-6">
                <div class="card">
                    <div class="card-body">
                        <h5 class="card-title">Current Balance</h5>
                        <p class="card-text">Balance Weight: <span class="balance-weight" id="currentBalance">
                                {{ number_format($plan->balance_weight, 3) }}</span> TON</p>
                    </div>
                </div>
            </div>
            <div class="col-md-6">
                <div class="card">
                    <div class="card-body">
                        <h5 class="card-title">Total Used Weight</h5>
                        <p class="card-text"><span id="totalUsedWeight">
                                {{ number_format($plan->total_used_weight, 3) }}</span> TON</p>
                    </div>
                </div>
            </div>
        </div>

        <!-- Process Sections -->
        <div id="process-sections">
            <div class="process-buttons">
                <button class="btn btn-success" id="initial-add-process">Add Process</button>
                <button class="btn btn-danger" id="deleteProcess">Delete Process</button>
            </div>

            <!-- Loop through processes -->
            @foreach ($plan->processes as $process)
                @if ($process->process_type == 'crs')
                    <div class="section-title" id="crs-title-{{ $process->process_id }}">
                        <h4>CRS – Cut to Width</h4>
                    </div>
                    <div class="process-section" id="crs-section-{{ $process->process_id }}"
                        data-process-id="{{ $process->process_id }}" data-process-type="crs">
                        <div class="row mb-3">
                            <div class="col-md-4">
                                <label for="crsWidth-{{ $process->process_id }}" class="form-label">Cut Width
                                    (mm)
                                </label>
                                <input type="number" class="form-control" id="crsWidth-{{ $process->process_id }}"
                                    placeholder="Enter width">
                            </div>
                            <div class="col-md-4">
                                <label for="crsQuantity-{{ $process->process_id }}" class="form-label">Quantity</label>
                                <input type="number" class="form-control" id="crsQuantity-{{ $process->process_id }}"
                                    placeholder="Enter quantity">
                            </div>
                            <div class="col-md-4 d-flex align-items-end">
                                <button class="btn btn-primary w-100" onclick="addCrsCut({{ $process->process_id }})">Add
                                    Cut</button>
                            </div>
                        </div>
                        <table class="table table-bordered">
                            <thead>
                                <tr>
                                    <th>Width (mm)</th>
                                    <th>Quantity</th>
                                    <th>Weight (TON)</th>
                                    <th>Action</th>
                                </tr>
                            </thead>
                            <tbody id="crsCutsTable-{{ $process->process_id }}">
                                @if (empty($crs_coil))
                                    <tr>
                                        <td colspan="4">No CRS cuts found.</td>
                                    </tr>
                                @else
                                    @foreach ($crs_coil as $index => $cut)
                                        @if ($cut->process_id == $process->process_id)
                                            <tr>
                                                <td>{{ number_format($cut->cut_width, 1) }}</td>
                                                <td>{{ $cut->quantity }}</td>
                                                <td>{{ number_format($cut->weight, 3) }}</td>
                                                <td>
                                                    <button class="btn btn-sm btn-danger remove-cut"
                                                        data-process-id="{{ $process->process_id }}"
                                                        data-index="{{ $index }}">Remove</button>
                                                </td>
                                            </tr>
                                        @endif
                                    @endforeach
                                @endif
                            </tbody>
                            <tfoot>
                                <tr>
                                    <th colspan="2">Total Weight</th>
                                    <th id="crsTotalWeight-{{ $process->process_id }}">0.000</th>
                                    <th></th>
                                </tr>
                            </tfoot>
                        </table>
                    </div>
                @elseif ($process->process_type == 'ctl')
                    <div class="section-title" id="ctl-title-{{ $process->process_id }}">
                        <h4>CTL – Cut to Length</h4>
                    </div>
                    <div class="process-section" id="ctl-section-{{ $process->process_id }}"
                        data-process-id="{{ $process->process_id }}" data-process-type="ctl">
                        <div class="alert alert-secondary mb-3">
                            <strong>CTL Process Rules:</strong><br>
                            - Independent when no CRS cuts (uses net coil width)<br>
                            - Dependent on CRS cuts when available<br>
                            - User inputs length and production weight; quantity calculated
                        </div>
                        <div class="row mb-3 dimension-select">
                            <div class="col-md-4">
                                <label for="ctlDependentCut-{{ $process->process_id }}" class="form-label">Select
                                    Dependent CRS Cut</label>
                                <select class="form-control" id="ctlDependentCut-{{ $process->process_id }}">
                                    <option value="">Select a CRS cut</option>
                                </select>
                            </div>
                            <div class="col-md-4">
                                <label for="ctlWidth-{{ $process->process_id }}" class="form-label">Width (mm)</label>
                                <input type="number" class="form-control" id="ctlWidth-{{ $process->process_id }}"
                                    readonly>
                            </div>
                            <div class="col-md-4">
                                <label for="ctlAvailableWeight-{{ $process->process_id }}" class="form-label">Available
                                    Weight (TON)</label>
                                <input type="number" class="form-control"
                                    id="ctlAvailableWeight-{{ $process->process_id }}" readonly>
                            </div>
                        </div>
                        <div class="row mb-3">
                            <div class="col-md-4">
                                <label for="ctlLength-{{ $process->process_id }}" class="form-label">Length
                                    (mm)</label>
                                <input type="number" class="form-control" id="ctlLength-{{ $process->process_id }}"
                                    placeholder="Enter length">
                            </div>
                            <div class="col-md-4">
                                <label for="ctlProductionWeight-{{ $process->process_id }}" class="form-label">Production
                                    Weight (TON)</label>
                                <input type="number" class="form-control"
                                    id="ctlProductionWeight-{{ $process->process_id }}" placeholder="Enter weight">
                            </div>
                            <div class="col-md-4 d-flex align-items-end">
                                <button class="btn btn-primary w-100" onclick="addCtlCut({{ $process->process_id }})">Add
                                    Cut</button>
                            </div>
                        </div>
                        <div class="row mb-3">
                            <div class="col-md-6">
                                <label class="form-label"><strong>Estimated Quantity:</strong></label>
                                <div class="form-control bg-light" id="previewCtlQuantity-{{ $process->process_id }}">0
                                </div>
                            </div>
                            <div class="col-md-6">
                                <label class="form-label"><strong>Estimated Used Weight (TON):</strong></label>
                                <div class="form-control bg-light" id="previewActualWeight-{{ $process->process_id }}">
                                    0.000</div>
                            </div>
                        </div>
                        <table class="table table-bordered">
                            <thead>
                                <tr>
                                    <th>Width (mm)</th>
                                    <th>Length (mm)</th>
                                    <th>Quantity</th>
                                    <th>Weight (TON)</th>
                                    <th>Action</th>
                                </tr>
                            </thead>
                            <tbody id="ctlCutsTable-{{ $process->process_id }}">
                                @if (empty($ctl_coil))
                                    <tr>
                                        <td colspan="5">No CTL cuts found.</td>
                                    </tr>
                                @else
                                    @foreach ($ctl_coil as $index => $cut)
                                        @if ($cut->process_id == $process->process_id)
                                            <tr>
                                                <td>{{ number_format($cut->cut_width, 1) }}</td>
                                                <td>{{ number_format($cut->cut_length, 1) }}</td>
                                                <td>{{ $cut->quantity }}</td>
                                                <td>{{ number_format($cut->weight, 3) }}</td>
                                                <td>
                                                    <button class="btn btn-sm btn-danger remove-ctl-cut"
                                                        data-process-id="{{ $process->process_id }}"
                                                        data-index="{{ $index }}">Remove</button>
                                                </td>
                                            </tr>
                                        @endif
                                    @endforeach
                                @endif
                            </tbody>
                            <tfoot>
                                <tr>
                                    <th colspan="3">Total Weight</th>
                                    <th id="ctlTotalWeight-{{ $process->process_id }}">0.000</th>
                                    <th></th>
                                </tr>
                            </tfoot>
                        </table>
                    </div>
                @endif
            @endforeach
        </div>

        <!-- Plan Summary -->
        <div class="section-title mt-5">
            <h4>Plan Summary</h4>
        </div>
        <div class="row">
            <div class="col-md-6">
                <div class="card">
                    <div class="card-body">
                        <h5 class="card-title">Material Utilization</h5>
                        <div class="progress mb-3">
                            <div class="progress-bar bg-success" role="progressbar" id="usedBar"
                                style="width: {{ $plan->coil->net_weight > 0 ? ($plan->total_used_weight / $plan->coil->net_weight) * 100 : 0 }}%"
                                aria-label="Used Weight">
                                {{ number_format($plan->coil->net_weight > 0 ? ($plan->total_used_weight / $plan->coil->net_weight) * 100 : 0, 1) }}%
                            </div>
                        </div>
                        <p class="card-text">Used: <span
                                id="summaryUsed">{{ number_format($plan->total_used_weight, 3) }}</span> TON</p>
                        <p class="card-text">Remaining: <span
                                id="summaryRemaining">{{ number_format($plan->balance_weight, 3) }}</span> TON</p>
                    </div>
                </div>
            </div>
            <div class="col-md-6">
                <div class="card">
                    <div class="card-body">
                        <h5 class="card-title">Plan Actions</h5>
                        <button class="btn btn-success w-100 mb-2" id="updatePlan">Update Plan</button>
                        <a href="{{ route('plans.index') }}" class="btn btn-primary w-100 mb-2">View Plans</a>
                        <button class="btn btn-secondary w-100" id="printPlan">Print Plan</button>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <!-- Add Process Modal -->
    <div class="modal fade" id="addProcessModal" tabindex="-1" aria-labelledby="addProcessModalLabel"
        aria-hidden="true">
        <div class="modal-dialog">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title" id="addProcessModalLabel">Add New Process</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
                </div>
                <div class="modal-body">
                    <div class="mb-3">
                        <label for="newProcessType" class="form-label">Select Process Type</label>
                        <select class="form-select" id="newProcessType">
                            <option value="">Select Process Type</option>
                            <option value="crs">CRS – Cut to Width</option>
                            <option value="ctl">CTL – Cut to Length</option>
                        </select>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                    <button type="button" class="btn btn-primary" id="confirmAddProcess">Add Process</button>
                </div>
            </div>
        </div>
    </div>

    <!-- Delete Process Modal -->
    <div class="modal fade" id="deleteProcessModal" tabindex="-1" aria-labelledby="deleteProcessModalLabel"
        aria-hidden="true">
        <div class="modal-dialog">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title" id="deleteProcessModalLabel">Delete a Process</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
                </div>
                <div class="modal-body">
                    <div class="mb-3">
                        <label for="deleteProcessId" class="form-label">Select Process to Delete</label>
                        <select class="form-select" id="deleteProcessId">
                            <option value="">Select a process</option>
                        </select>
                        <div id="noProcessesMessage" class="text-danger mt-2" style="display: none;">No processes
                            available to delete.</div>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                    <button type="button" class="btn btn-danger" id="confirmDeleteProcess">Delete Process</button>
                </div>
            </div>
        </div>
    </div>

<script src="https://code.jquery.com/jquery-3.6.0.min.js"></script>
<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
<script>
    function formatWeight(value) {
        return Number(value).toFixed(3);
    }

    let netWeight = {{ $plan->coil->net_weight ?? 0 }},
        width = {{ $plan->coil->width ?? 0 }},
        thickness = {{ $plan->coil->thickness ?? 0 }},
        density = {{ $plan->coil->material_type_density ?? 0 }},
        clientName = '{{ $plan->client->client_name ?? '' }}',
        crrNo = '{{ $plan->coil->crr_no ?? '' }}',
        grade = '{{ $plan->grade->grade_name ?? '' }}',
        balanceWeight = {{ $plan->balance_weight ?? 0 }},
        totalUsedWeight = {{ $plan->total_used_weight ?? 0 }},
        processes = [],
        processCounter = {{ $plan->processes->max('process_id') + 1 ?? 1 }},
        coilLength = 0;

    // Initialize processes with saved cuts and validate dependentIndex
    @foreach ($plan->processes as $process)
        processes.push({
            id: {{ $process->process_id }},
            type: '{{ $process->process_type }}',
            cuts: [
                @if ($process->process_type == 'crs')
                    @foreach ($crs_coil as $index => $cut)
                        @if ($cut->process_id == $process->process_id)
                            {
                                width: {{ $cut->cut_width ?? 0 }},
                                quantity: {{ $cut->quantity ?? 0 }},
                                weight: '{{ number_format($cut->weight ?? 0, 3) }}',
                                remainingWeight: '{{ number_format($cut->bal_weight ?? ($cut->weight ?? 0), 3) }}'
                            },
                        @endif
                    @endforeach
                @else
                    @foreach ($ctl_coil as $index => $cut)
                        @if ($cut->process_id == $process->process_id)
                            {
                                dependentIndex: (function() {
                                    const depIndex = '{{ $cut->dependent_index ?? '-1' }}';
                                    if (depIndex === '-1') return '-1';
                                    const [procId, cutIdx] = depIndex.split('-').map(Number);
                                    const crsProc = processes.find(p => p.id === procId && p.type === 'crs');
                                    if (crsProc && crsProc.cuts[cutIdx]) {
                                        return depIndex;
                                    }
                                    console.warn(`Invalid dependentIndex ${depIndex} for CTL cut in process {{ $process->process_id }}. Setting to '-1'.`);
                                    return '-1';
                                })(),
                                width: {{ $cut->cut_width ?? 0 }},
                                length: {{ $cut->cut_length ?? 0 }},
                                quantity: {{ $cut->quantity ?? 0 }},
                                weight: '{{ number_format($cut->weight ?? 0, 3) }}'
                            },
                        @endif
                    @endforeach
                @endif
            ].filter(cut => cut !== undefined) // Ensure no undefined cuts
        });
    @endforeach

    console.log('Initialized processes:', JSON.parse(JSON.stringify(processes)));

    function getCoilLengthFromWeight(weightTon, coilWidth, coilThickness, density) {
        const weightKg = Number(formatWeight(weightTon * 1000));
        if (density <= 0) return 0;
        const volume_mm3 = Number(formatWeight((weightKg / density) * 1000));
        if (coilWidth * coilThickness <= 0) return 0;
        return Number(formatWeight(volume_mm3 / (coilWidth * coilThickness)));
    }

    function calculateWeight(cutWidth, cutThickness, cutDensity, cutLength) {
        if (cutLength <= 0 || cutDensity <= 0) return 0;
        const volume = Number(formatWeight(cutWidth * cutThickness * cutLength));
        const weightKg = Number(formatWeight((volume * cutDensity) / 1000));
        return Number(formatWeight(weightKg / 1000));
    }

    function addCrsCut(processId) {
        const cutWidth = parseFloat($(`#crsWidth-${processId}`).val());
        const quantity = parseInt($(`#crsQuantity-${processId}`).val());
        if (isNaN(cutWidth) || cutWidth <= 0 || isNaN(quantity) || quantity <= 0) {
            alert('Enter valid width and quantity');
            return;
        }
        const usedWidth = processes.filter(p => p.type === 'crs').flatMap(p => p.cuts || []).reduce((sum, cut) => sum + cut.width * cut.quantity, 0);
        const availableWidth = width - usedWidth;
        if (cutWidth * quantity > availableWidth) {
            alert(`Total cut width (${(cutWidth * quantity).toFixed(1)}mm) cannot exceed available width (${availableWidth.toFixed(1)}mm)`);
            return;
        }
        let currentLength = coilLength;
        if (processes.some(p => p.type === 'ctl')) {
            currentLength = getCoilLengthFromWeight(balanceWeight, width, thickness, density);
        }
        const cutWeight = Number(formatWeight(calculateWeight(cutWidth, thickness, density, currentLength) * quantity));
        if (cutWeight > balanceWeight) {
            alert(`Insufficient balance weight (${balanceWeight.toFixed(3)} TON). Adding this cut (${cutWeight.toFixed(3)} TON) would result in a negative balance.`);
            return;
        }
        const process = processes.find(p => p.id === processId);
        process.cuts = process.cuts || [];
        process.cuts.push({
            width: cutWidth,
            quantity,
            weight: formatWeight(cutWeight),
            remainingWeight: formatWeight(cutWeight)
        });
        totalUsedWeight = Number(formatWeight(totalUsedWeight + cutWeight));
        balanceWeight = Number(formatWeight(netWeight - totalUsedWeight));
        updateUsedWeight();
        refreshCrsCutsTable(processId);
        refreshCrsDependentOptions();
        $(`#crsWidth-${processId}, #crsQuantity-${processId}`).val('');
    }

    function addCtlCut(processId) {
        const dependentIndex = $(`#ctlDependentCut-${processId}`).val();
        const length = parseFloat($(`#ctlLength-${processId}`).val());
        const productionWeight = parseFloat($(`#ctlProductionWeight-${processId}`).val());
        if (!dependentIndex || isNaN(length) || length <= 0 || isNaN(productionWeight) || productionWeight <= 0) {
            alert('Enter valid CTL values');
            return;
        }
        let cutWidth, availableWeight;
        if (dependentIndex === "-1") {
            cutWidth = width;
            const usedWeight = processes
                .filter(p => p.type === 'ctl')
                .flatMap(p => p.cuts || [])
                .filter(cut => cut && cut.dependentIndex === "-1")
                .reduce((s, c) => s + Number(c.weight), 0);
            availableWeight = Number(formatWeight(netWeight - usedWeight));
        } else {
            const [crsProcessId, cutIndex] = dependentIndex.split('-').map(Number);
            const crsProcess = processes.find(p => p.id === crsProcessId);
            const selectedCut = crsProcess.cuts[cutIndex];
            cutWidth = selectedCut.width;
            availableWeight = Number(formatWeight(selectedCut.remainingWeight));
        }
        if (productionWeight > availableWeight) {
            alert(`Exceeds available weight: ${availableWeight.toFixed(3)} TON`);
            return;
        }
        const pieceVol = Number(formatWeight(cutWidth * thickness * length));
        const pieceWeight_grams = Number(formatWeight(pieceVol * density / 1000));
        const pieceWeight_TON = Number(formatWeight(pieceWeight_grams / 1_000_000));
        const quantity = Math.floor(productionWeight / pieceWeight_TON);
        const actualWeight = Number(formatWeight(quantity * pieceWeight_TON));
        if (quantity <= 0) {
            alert('Production weight too low for one piece');
            return;
        }
        if (dependentIndex === "-1" && actualWeight > balanceWeight) {
            alert(`Insufficient balance weight (${balanceWeight.toFixed(3)} TON). Adding this cut (${actualWeight.toFixed(3)} TON) would result in a negative balance.`);
            return;
        }
        const process = processes.find(p => p.id === processId);
        process.cuts = process.cuts || [];
        process.cuts.push({
            dependentIndex,
            width: cutWidth,
            length,
            quantity,
            weight: formatWeight(actualWeight)
        });
        if (dependentIndex !== "-1") {
            const [crsProcessId, cutIndex] = dependentIndex.split('-').map(Number);
            const crsProcess = processes.find(p => p.id === crsProcessId);
            crsProcess.cuts[cutIndex].remainingWeight = formatWeight(Number(crsProcess.cuts[cutIndex].remainingWeight) - actualWeight);
        } else {
            totalUsedWeight = Number(formatWeight(totalUsedWeight + actualWeight));
        }
        balanceWeight = Number(formatWeight(netWeight - totalUsedWeight));
        updateUsedWeight();
        refreshCtlCutsTable(processId);
        $(`#ctlLength-${processId}, #ctlProductionWeight-${processId}`).val('');
        updateCtlDependentFields(processId);
        updateCtlPreview(processId);
    }

    function validateCoilDetails() {
        let isValid = true;
        const clientId = $('#client_id').val();
        const crrNoVal = $('#crr_no').val();
        const widthVal = parseFloat($('#width').val());
        const thicknessVal = parseFloat($('#thickness').val());
        const densityVal = parseFloat($('#density').val());
        const netWeightVal = parseFloat($('#netWeight').val());
        const gradeVal = $('#grade_id').val();
        $('#clientNameError').toggle(!clientId);
        $('#crrNoError').toggle(!crrNoVal);
        $('#widthError').toggle(isNaN(widthVal) || widthVal <= 0);
        $('#thicknessError').toggle(isNaN(thicknessVal) || thicknessVal <= 0);
        $('#densityError').toggle(isNaN(densityVal) || densityVal <= 0);
        $('#netWeightError').toggle(isNaN(netWeightVal) || netWeightVal <= 0);
        $('#gradeError').toggle(!gradeVal);
        isValid = clientId && crrNoVal && !isNaN(widthVal) && widthVal > 0 && !isNaN(thicknessVal) &&
            thicknessVal > 0 && !isNaN(densityVal) && densityVal > 0 && !isNaN(netWeightVal) && netWeightVal > 0 &&
            gradeVal;
        return isValid;
    }

    function setCoilDetails() {
        if (!validateCoilDetails()) return false;
        clientName = $('#client_id option:selected').text();
        crrNo = $('#crr_no option:selected').text();
        grade = $('#grade_id option:selected').text();
        netWeight = parseFloat($('#netWeight').val()) || 0;
        width = parseFloat($('#width').val()) || 0;
        thickness = parseFloat($('#thickness').val()) || 0;
        density = parseFloat($('#density').val()) || 0;
        coilLength = getCoilLengthFromWeight(netWeight, width, thickness, density);
        balanceWeight = netWeight - totalUsedWeight;
        $('#netWeight').val(formatWeight(netWeight));
        $('#width').val(formatWeight(width));
        $('#thickness').val(thickness);
        $('#density').val(density);
        updateBalance();
        return true;
    }

    function updateSelectedClientValues() {
        const clientId = $('#client_id').val();
        const crrNoDropdown = $('#crr_no');
        crrNoDropdown.empty().append('<option value="">Select CRR No.</option>');
        $('#thickness, #width, #grade_id, #netWeight, #inward_id, #density').val('');
        $('#currentBalance, #totalUsedWeight').text('0.000');
        $('#gradeError, #widthError, #thicknessError, #densityError, #netWeightError').hide();
        if (!clientId) {
            console.log('No client selected');
            return;
        }
        console.log('Client selected:', clientId);
        const url = '{{ route('getClientinfoInward', [':clientId']) }}'.replace(':clientId', clientId);
        $.ajax({
            url: url,
            type: 'GET',
            dataType: 'json',
            success: function(data) {
                console.log('Coils fetched:', data);
                if (!data || data.length === 0) {
                    alert('No coils found for the selected client.');
                    return;
                }
                data.forEach(function(coil) {
                    crrNoDropdown.append(`<option value="${coil.id}">${coil.crr_no}</option>`);
                });
                const planCoilId = '{{ $plan->coil_id ?? '' }}';
                if (planCoilId && crrNoDropdown.find(`option[value="${planCoilId}"]`).length) {
                    crrNoDropdown.val(planCoilId).trigger('change');
                }
            },
            error: function(xhr) {
                console.error('Error fetching coils:', xhr.status, xhr.responseText);
                alert('Failed to load coil information: ' + (xhr.responseJSON?.message || 'Unknown error'));
            }
        });
    }

    function getInfo(coilId) {
        if (!coilId) {
            console.log('No CRR No. selected');
            $('#grade_id, #thickness, #width, #density, #netWeight, #inward_id').val('');
            $('#gradeError').text('Please select a CRR No.').show();
            return;
        }
        if (processes.length > 0 && !confirm('Changing CRR No. will reset the plan. Proceed?')) {
            $('#crr_no').val('{{ $plan->coil_id ?? '' }}');
            return;
        }
        console.log('CRR No. change event triggered for coilId:', coilId);
        const url = '{{ route('getCrrinfoInward', [':coilId']) }}'.replace(':coilId', coilId);
        $.ajax({
            url: url,
            type: 'GET',
            dataType: 'json',
            success: function(data) {
                console.log('Full AJAX response:', data);
                if (!data || !data.crr_info) {
                    console.error('Invalid response: crr_info missing');
                    alert('Invalid coil data received.');
                    $('#grade_id, #thickness, #width, #density, #netWeight, #inward_id').val('');
                    $('#gradeError').text('Invalid coil data').show();
                    return;
                }
                const crrInfo = data.crr_info;
                const gradeId = crrInfo.grade_id || '';
                const gradeSelect = $('#grade_id');
                gradeSelect.val(gradeId);
                gradeSelect.prop('disabled', true);
                if (!gradeId) {
                    console.warn('Grade ID missing:', { grade_id: gradeId });
                    $('#gradeError').text('Grade data missing for this coil').show();
                } else {
                    $('#gradeError').hide();
                }
                $('#inward_id').val(crrInfo.inward_id || '');
                $('#thickness').val(crrInfo.thickness || '');
                $('#width').val(crrInfo.width || '');
                $('#density').val(crrInfo.material_type_density || '');
                $('#netWeight').val(formatWeight(crrInfo.net_weight || 0));
                localStorage.setItem('cdetail', JSON.stringify(crrInfo));
                if (!setCoilDetails()) {
                    console.error('Failed to validate coil details');
                    alert('Invalid coil details. Please check the selected coil.');
                }
            },
            error: function(xhr) {
                console.error('AJAX error fetching CRR info:', xhr.status, xhr.responseText);
                alert('Failed to load CRR information: ' + (xhr.responseJSON?.message || 'Unknown error'));
                $('#grade_id, #thickness, #width, #density, #netWeight, #inward_id').val('');
                $('#gradeError').text('Failed to load coil data').show();
            }
        });
    }

    function updateBalance() {
        $('#currentBalance').text(formatWeight(netWeight));
        $('#summaryRemaining').text(formatWeight(balanceWeight));
        const usedPercent = netWeight > 0 ? (totalUsedWeight / netWeight) * 100 : 0;
        $('#usedBar').css('width', usedPercent + '%').text(`Used: ${usedPercent.toFixed(1)}%`);
    }

    function updateUsedWeight() {
        $('#totalUsedWeight').text(formatWeight(totalUsedWeight));
        $('#summaryUsed').text(formatWeight(totalUsedWeight));
        updateBalance();
    }

    function refreshCrsDependentOptions() {
        try {
            processes.forEach(p => {
                if (p.type === 'ctl') {
                    const select = $(`#ctlDependentCut-${p.id}`);
                    select.empty().append('<option value="">Select a CRS cut</option>');
                    const hasCrsCuts = processes.some(pr => pr.type === 'crs' && pr.cuts && pr.cuts.length > 0);
                    if (!hasCrsCuts) {
                        const availableWeight = Number(formatWeight(netWeight - totalUsedWeight));
                        select.append(`<option value="-1">${width.toFixed(1)}mm (${availableWeight.toFixed(3)} TON)</option>`);
                        select.val('-1');
                        updateCtlDependentFields(p.id);
                    } else {
                        processes.forEach(pr => {
                            if (pr.type === 'crs' && pr.cuts) {
                                pr.cuts.forEach((cut, index) => {
                                    select.append(`<option value="${pr.id}-${index}">${cut.width.toFixed(1)}mm (${cut.remainingWeight}) TON</option>`);
                                });
                            }
                        });
                        select.val('');
                        updateCtlDependentFields(p.id);
                    }
                    select.prop('disabled', hasCrsCuts && !processes.some(pr => pr.type === 'crs' && pr.cuts && pr.cuts.length > 0));
                }
            });
        } catch (error) {
            console.error('Error in refreshCrsDependentOptions:', error);
            alert('Error updating dependent options. Check console for details.');
        }
    }

    function refreshCrsCutsTable(processId) {
        const tableBody = $(`#crsCutsTable-${processId}`);
        tableBody.empty();
        const process = processes.find(p => p.id === processId);
        let crsTotalWeight = 0;
        console.log(`Refreshing CRS cuts for process ${processId}`, process.cuts);
        if (process.cuts && process.cuts.length) {
            process.cuts.forEach((cut, index) => {
                crsTotalWeight = Number(formatWeight(crsTotalWeight + Number(cut.weight)));
                tableBody.append(`
                    <tr>
                        <td>${Number(cut.width).toFixed(1)}</td>
                        <td>${cut.quantity}</td>
                        <td>${cut.weight}</td>
                        <td><button class="btn btn-sm btn-danger remove-cut" data-process-id="${processId}" data-index="${index}">Remove</button></td>
                    </tr>
                `);
            });
        }
        $(`#crsTotalWeight-${processId}`).text(formatWeight(crsTotalWeight));
        if (!process.cuts || process.cuts.length === 0) {
            tableBody.append('<tr><td colspan="4">No CRS cuts added.</td></tr>');
        }
    }

    function removeCrsCut(processId, index) {
        console.log(`Starting removeCrsCut for Process ${processId}, Index ${index}`);
        try {
            const process = processes.find(p => p.id === processId);
            if (!process || !process.cuts || !process.cuts[index]) {
                console.error(`CRS process ${processId} or cut index ${index} not found.`);
                alert('Error: CRS cut not found.');
                return;
            }
            const cut = process.cuts[index];
            const widthToRestore = Number(cut.width) * Number(cut.quantity);
            let weightToRestore = Number(cut.weight);
            console.log(`CRS cut details: width=${cut.width}, quantity=${cut.quantity}, weight=${cut.weight}, widthToRestore=${widthToRestore}`);

            const dependentCtlCuts = [];
            processes
                .filter(p => p.type === 'ctl')
                .forEach(p => {
                    if (p.cuts) {
                        p.cuts.forEach((cut, cutIndex) => {
                            if (cut.dependentIndex === `${processId}-${index}`) {
                                dependentCtlCuts.push({ process: p, cut, cutIndex });
                            }
                        });
                    }
                });
            console.log(`Found ${dependentCtlCuts.length} dependent CTL cuts`);

            if (dependentCtlCuts.length) {
                if (!confirm(`This will also remove ${dependentCtlCuts.length} dependent CTL cut(s). Proceed?`)) {
                    console.log('User cancelled CRS cut removal.');
                    return;
                }
            }

            dependentCtlCuts.forEach(({ process, cut }) => {
                weightToRestore += Number(cut.weight);
                const cutIndex = process.cuts.findIndex(c => c === cut);
                if (cutIndex !== -1) {
                    process.cuts.splice(cutIndex, 1);
                    console.log(`Removed dependent CTL cut (Process ${process.id}, Cut ${cutIndex}) with weight ${cut.weight} TON`);
                }
            });

            process.cuts.splice(index, 1);
            console.log(`Removed CRS cut from Process ${processId}, Index ${index}`);

            processes
                .filter(p => p.type === 'ctl')
                .forEach(p => {
                    if (p.cuts) {
                        p.cuts.forEach(cut => {
                            if (cut.dependentIndex !== "-1") {
                                const [depProcessId, depCutIndex] = cut.dependentIndex.split('-').map(Number);
                                if (depProcessId === processId && depCutIndex > index) {
                                    cut.dependentIndex = `${processId}-${depCutIndex - 1}`;
                                    console.log(`Updated dependentIndex for CTL cut in Process ${p.id}: ${cut.dependentIndex}`);
                                }
                            }
                        });
                    }
                });

            width = Number(formatWeight(width + widthToRestore));
            totalUsedWeight = Number(formatWeight(totalUsedWeight - weightToRestore));
            netWeight = Number(formatWeight(netWeight + weightToRestore));
            balanceWeight = Number(formatWeight(netWeight - totalUsedWeight));
            console.log(`Updated values: width=${width}, netWeight=${netWeight}, totalUsedWeight=${totalUsedWeight}, balanceWeight=${balanceWeight}`);

            $('#width').val(formatWeight(width));
            $('#netWeight').val(formatWeight(netWeight));
            setCoilDetails();
            refreshCrsCutsTable(processId);
            refreshCrsDependentOptions();
            updateUsedWeight();

            processes.filter(p => p.type === 'ctl').forEach(p => {
                refreshCtlCutsTable(p.id);
                updateCtlDependentFields(p.id);
                updateCtlPreview(p.id);
            });

            console.log('UI updated successfully');
            console.log('Processes after removal:', JSON.parse(JSON.stringify(processes)));
        } catch (error) {
            console.error('Error in removeCrsCut:', error);
            alert('Error removing CRS cut. Check console for details.');
        }
    }

    function updateCtlDependentFields(processId) {
        try {
            const selectedIndex = $(`#ctlDependentCut-${processId}`).val();
            if (!selectedIndex) {
                $(`#ctlWidth-${processId}, #ctlAvailableWeight-${processId}`).val('');
                updateCtlPreview(processId);
                return;
            }
            let cutWidth, availableWeight;
            if (selectedIndex === "-1") {
                cutWidth = width;
                const usedWeight = processes
                    .filter(p => p.type === 'ctl' && p.cuts)
                    .flatMap(p => p.cuts)
                    .filter(cut => cut && cut.dependentIndex === "-1")
                    .reduce((s, c) => s + Number(c.weight), 0);
                availableWeight = Number(formatWeight(netWeight - usedWeight));
            } else {
                const [crsProcessId, cutIndex] = selectedIndex.split('-').map(Number);
                const crsProcess = processes.find(p => p.id === crsProcessId);
                if (!crsProcess || !crsProcess.cuts || !crsProcess.cuts[cutIndex]) {
                    console.error(`CRS process ${crsProcessId} or cut ${cutIndex} not found.`);
                    $(`#ctlWidth-${processId}, #ctlAvailableWeight-${processId}`).val('');
                    updateCtlPreview(processId);
                    return;
                }
                const selectedCut = crsProcess.cuts[cutIndex];
                cutWidth = selectedCut.width;
                availableWeight = Number(formatWeight(selectedCut.remainingWeight));
            }
            $(`#ctlWidth-${processId}`).val(cutWidth.toFixed(1));
            $(`#ctlAvailableWeight-${processId}`).val(formatWeight(availableWeight));
            updateCtlPreview(processId);
        } catch (error) {
            console.error(`Error in updateCtlDependentFields for process ${processId}:`, error);
            $(`#ctlWidth-${processId}, #ctlAvailableWeight-${processId}`).val('');
            updateCtlPreview(processId);
        }
    }

    function refreshCtlCutsTable(processId) {
        const tableBody = $(`#ctlCutsTable-${processId}`);
        tableBody.empty();
        const process = processes.find(p => p.id === processId);
        let ctlTotalWeight = 0;
        console.log(`Refreshing CTL cuts for process ${processId}`, process.cuts);
        if (process.cuts && process.cuts.length) {
            process.cuts.forEach((cut, index) => {
                ctlTotalWeight = Number(formatWeight(ctlTotalWeight + Number(cut.weight)));
                let depWidth = cut.width;
                if (cut.dependentIndex !== "-1") {
                    const [crsProcessId, cutIndex] = cut.dependentIndex.split('-').map(Number);
                    const crsProcess = processes.find(p => p.id === crsProcessId);
                    if (crsProcess && crsProcess.cuts && crsProcess.cuts[cutIndex]) {
                        depWidth = crsProcess.cuts[cutIndex].width;
                    } else {
                        console.warn(`Invalid dependentIndex ${cut.dependentIndex} in CTL cut (Process ${processId}, Index ${index}). Using cut width.`);
                    }
                }
                tableBody.append(`
                    <tr>
                        <td>${Number(depWidth).toFixed(1)}</td>
                        <td>${Number(cut.length).toFixed(1)}</td>
                        <td>${cut.quantity}</td>
                        <td>${cut.weight}</td>
                        <td><button class="btn btn-sm btn-danger remove-ctl-cut" data-process-id="${processId}" data-index="${index}">Remove</button></td>
                    </tr>
                `);
            });
        }
        $(`#ctlTotalWeight-${processId}`).text(formatWeight(ctlTotalWeight));
        if (!process.cuts || process.cuts.length === 0) {
            tableBody.append('<tr><td colspan="5">No CTL cuts added.</td></tr>');
        }
    }

    function removeCtlCut(processId, index) {
        console.log(`Starting removeCtlCut for Process ${processId}, Index ${index}`);
        try {
            const process = processes.find(p => p.id === processId);
            if (!process || !process.cuts || !process.cuts[index]) {
                console.error(`CTL process ${processId} or cut index ${index} not found.`);
                alert('Error: CTL cut not found.');
                return;
            }
            const cut = process.cuts[index];
            const weightToRestore = Number(cut.weight);

            if (cut.dependentIndex !== "-1") {
                // Dependent CTL cut: Restore weight to parent CRS cut's remainingWeight only
                const [crsProcessId, cutIndex] = cut.dependentIndex.split('-').map(Number);
                const crsProcess = processes.find(p => p.id === crsProcessId);
                if (crsProcess && crsProcess.cuts && crsProcess.cuts[cutIndex]) {
                    crsProcess.cuts[cutIndex].remainingWeight = formatWeight(
                        Number(crsProcess.cuts[cutIndex].remainingWeight) + weightToRestore
                    );
                    console.log(`Restored ${weightToRestore} TON to CRS cut (Process ${crsProcessId}, Cut ${cutIndex}). New remainingWeight: ${crsProcess.cuts[cutIndex].remainingWeight}`);
                } else {
                    console.warn(`CRS process ${crsProcessId} or cut ${cutIndex} not found for dependent CTL cut. Skipping weight restoration.`);
                    // Explicitly do not modify netWeight or totalUsedWeight
                }
                // Update balanceWeight
            balanceWeight = Number(formatWeight(netWeight));
            } else {
                // Independent CTL cut: Adjust totalUsedWeight and netWeight
                totalUsedWeight = Number(formatWeight(totalUsedWeight - weightToRestore));
                netWeight = Number(formatWeight(netWeight));
                console.log(`Independent CTL cut removed. Restored ${weightToRestore} TON to totalUsedWeight: ${totalUsedWeight}, netWeight: ${netWeight}`);
                // Update balanceWeight
            balanceWeight = 0;
            //netWeight =0;


            }

            // Remove the CTL cut
            process.cuts.splice(index, 1);

            

            // Update UI
            $('#netWeight').val(formatWeight(netWeight));
            $('#currentBalance').val(formatWeight(netWeight));
            setCoilDetails();
            updateUsedWeight();
            refreshCtlCutsTable(processId);
            updateCtlDependentFields(processId);
            updateCtlPreview(processId);
            refreshCrsDependentOptions();

            console.log(`After removing CTL cut - Process ${processId}, Index ${index}: netWeight=${netWeight}, totalUsedWeight=${totalUsedWeight}, balanceWeight=${balanceWeight}`);
            console.log('Processes after removal:', JSON.parse(JSON.stringify(processes)));
        } catch (error) {
            console.error('Error in removeCtlCut:', error);
            alert('Error removing CTL cut. Check console for details.');
        }
    }

    function resetPlan() {
        console.log('Starting resetPlan');
        try {
            processes.forEach(p => {
                if (p.type === 'crs') {
                    p.cuts.forEach(cut => cut.remainingWeight = formatWeight(cut.weight));
                    $(`#crsCutsTable-${p.id}`).empty();
                    $(`#crsTotalWeight-${p.id}`).text('0.000');
                    $(`#crsWidth-${p.id}, #crsQuantity-${p.id}`).val('');
                } else {
                    p.cuts = [];
                    $(`#ctlCutsTable-${p.id}`).empty();
                    $(`#ctlTotalWeight-${p.id}`).text('0.000');
                    $(`#ctlLength-${p.id}, #ctlProductionWeight-${p.id}, #ctlDependentCut-${p.id}, #ctlWidth-${p.id}, #ctlAvailableWeight-${p.id}`).val('');
                    $(`#previewCtlQuantity-${p.id}`).text('0');
                    $(`#previewActualWeight-${p.id}`).text('0.000');
                }
                $(`#${p.type}-title-${p.id}, #${p.type}-section-${p.id}`).remove();
            });
            processes = [];
            processCounter = 1;
            totalUsedWeight = 0;
            balanceWeight = Number(formatWeight(netWeight));
            setCoilDetails();
            updateUsedWeight();
            refreshCrsDependentOptions();
            $('#process-sections').append($('.process-buttons'));
            $('#newProcessType option[value="crs"]').prop('disabled', false);
            console.log('Plan reset successfully');
            console.log('Processes after reset:', JSON.parse(JSON.stringify(processes)));
        } catch (error) {
            console.error('Error in resetPlan:', error);
            alert('Error resetting plan. Check console for details.');
        }
    }

    function updateCtlPreview(processId) {
        const length = parseFloat($(`#ctlLength-${processId}`).val());
        const productionWeight = parseFloat($(`#ctlProductionWeight-${processId}`).val());
        const selectedIndex = $(`#ctlDependentCut-${processId}`).val();
        if (!selectedIndex || isNaN(length) || length <= 0 || isNaN(productionWeight) || productionWeight <= 0) {
            $(`#previewCtlQuantity-${processId}`).text('0');
            $(`#previewActualWeight-${processId}`).text('0.000');
            return;
        }
        let cutWidth;
        if (selectedIndex === "-1") {
            cutWidth = width;
        } else {
            const [crsProcessId, cutIndex] = selectedIndex.split('-').map(Number);
            const crsProcess = processes.find(p => p.id === crsProcessId);
            if (!crsProcess || !crsProcess.cuts || !crsProcess.cuts[cutIndex]) {
                $(`#previewCtlQuantity-${processId}`).text('0');
                $(`#previewActualWeight-${processId}`).text('0.000');
                return;
            }
            cutWidth = crsProcess.cuts[cutIndex].width;
        }
        const pieceVol = Number(formatWeight(cutWidth * thickness * length));
        const pieceWeight_grams = Number(formatWeight(pieceVol * density / 1000));
        const pieceWeight_TON = Number(formatWeight(pieceWeight_grams / 1_000_000));
        const quantity = Math.floor(productionWeight / pieceWeight_TON);
        const actualWeight = Number(formatWeight(quantity * pieceWeight_TON));
        $(`#previewCtlQuantity-${processId}`).text(quantity);
        $(`#previewActualWeight-${processId}`).text(formatWeight(actualWeight));
    }

    function addProcess(processType) {
        if (netWeight === 0 || !validateCoilDetails()) {
            alert('Please select valid coil details before adding a process.');
            return;
        }
        if (balanceWeight <= 0) {
            alert('Insufficient balance weight to add a new process.');
            return;
        }
        if (processType === 'crs' && processes.some(p => p.type === 'crs')) {
            alert('Only one CRS process is allowed per plan.');
            $('#newProcessType option[value="crs"]').prop('disabled', true);
            return;
        }
        const newProcessId = processCounter++;
        processes.push({
            id: newProcessId,
            type: processType,
            cuts: []
        });
        const sectionHtml = processType === 'crs' ? `
            <div class="section-title" id="crs-title-${newProcessId}"><h4>CRS – Cut to Width</h4></div>
            <div class="process-section" id="crs-section-${newProcessId}" data-process-id="${newProcessId}" data-process-type="crs">
                <div class="alert alert-secondary mb-3"><strong>CRS Process Rules:</strong><br>- User can cut coil vertically in multiple sizes<br>- Only one CRS process allowed</div>
                <div class="row mb-3">
                    <div class="col-md-4"><label for="crsWidth-${newProcessId}" class="form-label">Cut Width (mm)</label><input type="number" class="form-control" id="crsWidth-${newProcessId}" placeholder="Enter width"></div>
                    <div class="col-md-4"><label for="crsQuantity-${newProcessId}" class="form-label">Quantity</label><input type="number" class="form-control" id="crsQuantity-${newProcessId}" placeholder="Enter quantity"></div>
                    <div class="col-md-4 d-flex align-items-end"><button class="btn btn-primary w-100" onclick="addCrsCut(${newProcessId})">Add Cut</button></div>
                </div>
                <table class="table table-bordered">
                    <thead><tr><th>Width (mm)</th><th>Quantity</th><th>Weight (TON)</th><th>Action</th></tr></thead>
                    <tbody id="crsCutsTable-${newProcessId}"><tr><td colspan="4">No CRS cuts added.</td></tr></tbody>
                    <tfoot><tr><th colspan="2">Total Weight</th><th id="crsTotalWeight-${newProcessId}">0.000</th><th></th></tr></tfoot>
                </table>
            </div>
        ` : `
            <div class="section-title" id="ctl-title-${newProcessId}"><h4>CTL – Cut to Length</h4></div>
            <div class="process-section" id="ctl-section-${newProcessId}" data-process-id="${newProcessId}" data-process-type="ctl">
                <div class="alert alert-secondary mb-3"><strong>CTL Process Rules:</strong><br>- Independent when no CRS cuts (uses net coil width)<br>- Dependent on CRS cuts when available<br>- User inputs length and production weight; quantity calculated</div>
                <div class="row mb-3 dimension-select">
                    <div class="col-md-4"><label for="ctlDependentCut-${newProcessId}" class="form-label">Select Dependent CRS Cut</label><select class="form-control" id="ctlDependentCut-${newProcessId}"></select></div>
                    <div class="col-md-4"><label for="ctlWidth-${newProcessId}" class="form-label">Width (mm)</label><input type="number" class="form-control" id="ctlWidth-${newProcessId}" readonly></div>
                    <div class="col-md-4"><label for="ctlAvailableWeight-${newProcessId}" class="form-label">Available Weight (TON)</label><input type="number" class="form-control" id="ctlAvailableWeight-${newProcessId}" readonly></div>
                </div>
                <div class="row mb-3">
                    <div class="col-md-4"><label for="ctlLength-${newProcessId}" class="form-label">Length (mm)</label><input type="number" class="form-control" id="ctlLength-${newProcessId}" placeholder="Enter length"></div>
                    <div class="col-md-4"><label for="ctlProductionWeight-${newProcessId}" class="form-label">Production Weight (TON)</label><input type="number" class="form-control" id="ctlProductionWeight-${newProcessId}" placeholder="Enter weight"></div>
                    <div class="col-md-4 d-flex align-items-end"><button class="btn btn-primary w-100" onclick="addCtlCut(${newProcessId})">Add Cut</button></div>
                </div>
                <div class="row mb-3">
                    <div class="col-md-6"><label class="form-label"><strong>Estimated Quantity:</strong></label><div class="form-control bg-light" id="previewCtlQuantity-${newProcessId}">0</div></div>
                    <div class="col-md-6"><label class="form-label"><strong>Estimated Used Weight (TON):</strong></label><div class="form-control bg-light" id="previewActualWeight-${newProcessId}">0.000</div></div>
                </div>
                <table class="table table-bordered">
                    <thead><tr><th>Width (mm)</th><th>Length (mm)</th><th>Quantity</th><th>Weight (TON)</th><th>Action</th></tr></thead>
                    <tbody id="ctlCutsTable-${newProcessId}"><tr><td colspan="5">No CTL cuts added.</td></tr></tbody>
                    <tfoot><tr><th colspan="3">Total Weight</th><th id="ctlTotalWeight-${newProcessId}">0.000</th><th></th></tr></tfoot>
                </table>
            </div>
        `;
        const lastProcessSection = $('#process-sections').find('.process-section').last();
        if (lastProcessSection.length) {
            lastProcessSection.after(sectionHtml);
        } else {
            $('.process-buttons').before(sectionHtml);
        }
        $('#process-sections').append($('.process-buttons'));
        if (processType === 'ctl') {
            $(`#ctlDependentCut-${newProcessId}`).on('change', () => updateCtlDependentFields(newProcessId));
            $(`#ctlLength-${newProcessId}, #ctlProductionWeight-${newProcessId}, #ctlDependentCut-${newProcessId}`).on('input change', () => updateCtlPreview(newProcessId));
        }
        refreshCrsDependentOptions();
        if (processType === 'ctl') updateCtlDependentFields(newProcessId);
        $('#addProcessModal').modal('hide');
        $('#newProcessType option[value="crs"]').prop('disabled', processes.some(p => p.type === 'crs'));
    }

    function deleteProcess(processId) {
        console.log(`Starting deleteProcess for Process ${processId}`);
        try {
            const process = processes.find(p => p.id === parseInt(processId));
            if (!process) {
                console.error(`Process ${processId} not found.`);
                alert('Error: Process not found.');
                return;
            }
            let weightToRemove = 0;
            let widthToRestore = 0;
            if (process.type === 'crs') {
                process.cuts = process.cuts || [];
                process.cuts.forEach(cut => {
                    weightToRemove += Number(cut.weight);
                    widthToRestore += Number(cut.width) * Number(cut.quantity);
                });
                const dependentCtlProcesses = processes.filter(p => p.type === 'ctl' && p.cuts && p.cuts.some(cut => cut.dependentIndex.split('-')[0] == processId));
                if (dependentCtlProcesses.length && !confirm(`This will also remove ${dependentCtlProcesses.length} dependent CTL processes and all their cuts. Proceed?`)) {
                    console.log('User cancelled process deletion.');
                    return;
                }
                dependentCtlProcesses.forEach(depProcess => {
                    depProcess.cuts = depProcess.cuts || [];
                    depProcess.cuts.forEach(cut => {
                        weightToRemove += Number(cut.weight);
                    });
                    processes = processes.filter(p => p.id !== depProcess.id);
                    $(`#ctl-title-${depProcess.id}, #ctl-section-${depProcess.id}`).remove();
                    console.log(`Removed dependent CTL process ${depProcess.id} with ${depProcess.cuts.length} cuts`);
                });
                totalUsedWeight = Number(formatWeight(totalUsedWeight - weightToRemove));
                netWeight = Number(formatWeight(netWeight + weightToRemove));
                width = Number(formatWeight(width + widthToRestore));
            } else {
                process.cuts = process.cuts || [];
                process.cuts.forEach(cut => {
                    if (cut.dependentIndex !== "-1") {
                        const [crsProcessId, cutIndex] = cut.dependentIndex.split('-').map(Number);
                        const crsProcess = processes.find(p => p.id === crsProcessId);
                        if (crsProcess && crsProcess.cuts && crsProcess.cuts[cutIndex]) {
                            crsProcess.cuts[cutIndex].remainingWeight = formatWeight(
                                Number(crsProcess.cuts[cutIndex].remainingWeight) + Number(cut.weight)
                            );
                            console.log(`Restored ${cut.weight} TON to CRS cut (Process ${crsProcessId}, Cut ${cutIndex})`);
                        } else {
                            weightToRemove += Number(cut.weight);
                        }
                    } else {
                        weightToRemove += Number(cut.weight);
                    }
                });
                totalUsedWeight = Number(formatWeight(totalUsedWeight - weightToRemove));
                netWeight = Number(formatWeight(netWeight + weightToRemove));
            }
            processes = processes.filter(p => p.id !== parseInt(processId));
            $(`#${process.type}-title-${processId}, #${process.type}-section-${processId}`).remove();
            balanceWeight = Number(formatWeight(netWeight - totalUsedWeight));
            $('#netWeight').val(formatWeight(netWeight));
            $('#width').val(formatWeight(width));
            setCoilDetails();
            updateUsedWeight();
            processes.forEach(p => p.cuts = p.cuts || []);
            refreshCrsDependentOptions();
            $('#process-sections').append($('.process-buttons'));
            console.log('After deletion - width:', width, 'netWeight:', netWeight, 'totalUsedWeight:', totalUsedWeight, 'balanceWeight:', balanceWeight);
            console.log('Processes after deletion:', JSON.parse(JSON.stringify(processes)));
        } catch (error) {
            console.error('Error in deleteProcess:', error);
            alert('Error deleting process. Check console for details.');
        }
    }

    function populateDeleteProcessModal() {
        const select = $('#deleteProcessId');
        select.empty().append('<option value="">Select a process</option>');
        if (processes.length === 0) {
            $('#noProcessesMessage').show();
            $('#confirmDeleteProcess').prop('disabled', true);
        } else {
            processes.forEach(p => {
                select.append(`<option value="${p.id}">${p.type.toUpperCase()} Process ${p.id}</option>`);
            });
            $('#noProcessesMessage').hide();
            $('#confirmDeleteProcess').prop('disabled', false);
        }
    }

    $(document).ready(function() {
        setCoilDetails();
        coilLength = getCoilLengthFromWeight(netWeight, width, thickness, density);
        refreshCrsDependentOptions();
        updateUsedWeight();
        $('#client_id').on('change', updateSelectedClientValues);
        $('#crr_no').on('change', function() {
            getInfo($(this).val());
        });
        $('#initial-add-process').on('click', function() {
            $('#addProcessModal').modal('show');
        });
        $('#confirmAddProcess').on('click', function() {
            const processType = $('#newProcessType').val();
            if (!processType) {
                alert('Please select a process type.');
                return;
            }
            addProcess(processType);
        });
        $('#deleteProcess').on('click', function() {
            populateDeleteProcessModal();
            $('#deleteProcessModal').modal('show');
        });
        $('#confirmDeleteProcess').on('click', function() {
            const processId = $('#deleteProcessId').val();
            if (!processId) {
                alert('Please select a process to delete.');
                return;
            }
            deleteProcess(processId);
            $('#deleteProcessModal').modal('hide');
        });
        $(document).on('click', '.remove-cut', function() {
            const processId = $(this).data('process-id');
            const index = $(this).data('index');
            removeCrsCut(processId, index);
        });
        $(document).on('click', '.remove-ctl-cut', function() {
            const processId = $(this).data('process-id');
            const index = $(this).data('index');
            removeCtlCut(processId, index);
        });
        $('#updatePlan').on('click', function() {
            if (!validateCoilDetails()) {
                alert('Please ensure all coil details are valid.');
                return;
            }
            if (processes.length === 0) {
                alert('At least one process is required.');
                return;
            }
            const planData = {
                plan_id: {{ $plan->id }},
                client_id: $('#client_id').val(),
                coil_id: $('#crr_no').val(),
                grade_id: $('#grade_id').val(),
                inward_id: $('#inward_id').val(),
                width: parseFloat($('#width').val()),
                thickness: parseFloat($('#thickness').val()),
                density: parseFloat($('#density').val()),
                net_weight: parseFloat($('#netWeight').val()),
                total_used_weight: totalUsedWeight,
                balance_weight: balanceWeight,
                processes: processes.map(p => ({
                    type: p.type,
                    cuts: (p.cuts || []).map(cut => ({
                        width: Number(cut.width.toFixed(2)),
                        quantity: cut.quantity,
                        weight: cut.weight,
                        length: cut.length ? Number(cut.length.toFixed(2)) : null,
                        dependent_index: cut.dependentIndex || null,
                        remainingWeight: cut.remainingWeight
                    }))
                }))
            };
            console.log('Submitting plan data:', planData);
            $.ajax({
                url: '{{ route('plans.update', $plan->id) }}',
                type: 'PUT',
                data: JSON.stringify(planData),
                contentType: 'application/json',
                headers: {
                    'X-CSRF-TOKEN': $('meta[name="csrf-token"]').attr('content')
                },
                success: function(response) {
                    alert('Plan updated successfully!');
                    window.location.href = '{{ route('plans.index') }}';
                },
                error: function(xhr) {
                    console.error('Error updating plan:', xhr.status, xhr.responseText);
                    alert('Failed to update plan: ' + (xhr.responseJSON?.message || 'Unknown error'));
                }
            });
        });
        $('#printPlan').on('click', function() {
            window.print();
        });
        processes.forEach(p => {
            p.cuts = p.cuts || [];
            if (p.type === 'ctl') {
                $(`#ctlDependentCut-${p.id}`).on('change', () => updateCtlDependentFields(p.id));
                $(`#ctlLength-${p.id}, #ctlProductionWeight-${p.id}, #ctlDependentCut-${p.id}`).on('input change', () => updateCtlPreview(p.id));
                updateCtlDependentFields(p.id);
            }
            if (p.type === 'crs') {
                refreshCrsCutsTable(p.id);
            } else {
                refreshCtlCutsTable(p.id);
            }
        });
        if ($('#client_id').val()) {
            updateSelectedClientValues();
        }
    });
</script>
@endsection